/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.action.client;

import webwork.action.Action;

import java.io.*;
import java.net.URL;
import java.net.URLConnection;
import java.applet.Applet;

/**
 * The ClientDispatcher is used by remote clients to execute actions on a server running WebWork.
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 */
public class ClientDispatcher
{
    // Attributes ----------------------------------------------------
    String host;

    // Static --------------------------------------------------------
    /**
     * Standard dispatcher servlet URL
     */
    public static final String DISPATCHER = "/servlet/dispatch";

    // Constructors --------------------------------------------------
    /**
     * Create a new dispatcher with an applet providing the URL.
     * <p/>
     * The servlet needs to be deployed with the name "dispatch".
     */
    public ClientDispatcher(Applet anApplet, String context)
            throws java.net.MalformedURLException
    {
        this(new URL(anApplet.getDocumentBase(),
                context + DISPATCHER).toExternalForm());
    }

    /**
     * Create a new dispatcher.
     *
     * @param aHost The URL of the servlet handler
     */
    public ClientDispatcher(String aHost)
    {
        host = aHost;
    }

    // Public --------------------------------------------------------
    public String getHost()
    {
        return host;
    }

    /**
     * Execute the action on the server.
     *
     * @return the result of the execution, including the action itself
     */
    public ActionResult execute(Action anAction)
            throws Exception
    {
        URL webworkHostServlet = new URL(host);
        URLConnection webworkHostServletConnection = webworkHostServlet.openConnection();

        // Inform the connection that we will send output and accept input
        webworkHostServletConnection.setDoInput(true);
        webworkHostServletConnection.setDoOutput(true);

        // Don't use a cached version of URL connection.
        webworkHostServletConnection.setUseCaches(false);

        // Specify the content type that we will send binary data
        webworkHostServletConnection.setRequestProperty("Content-Type", "application/octet-stream");

        // Send the action to the servlet using serialization
        ObjectOutputStream out = createObjectOutputStream(new BufferedOutputStream(webworkHostServletConnection.getOutputStream()));

        // Serialize the action
        out.writeObject(anAction);
        out.flush();

        // Get the result
        ObjectInputStream in = createObjectInputStream(new BufferedInputStream(webworkHostServletConnection.getInputStream()));

        ActionResult actionResult = (ActionResult) in.readObject();

        out.close();
        in.close();

        if (actionResult.getException() != null)
        {
            throw actionResult.getException();
        }

        return actionResult;
    }

    // Protected -----------------------------------------------------
    /**
     * Create an object input stream that wraps the URL connection stream.
     * <p/>
     * This method can be overriden in order to create custom streams.
     *
     * @param in the underlying stream. It is buffered
     *
     * @return an object input stream
     *
     * @throws IOException
     */
    protected ObjectInputStream createObjectInputStream(InputStream in)
            throws IOException
    {
        return new ObjectInputStream(in);
    }

    /**
     * Create an object input stream that wraps the URL connection stream.
     * <p/>
     * This method can be overriden in order to create custom streams.
     *
     * @param out The OutputStream to wrap
     *
     * @return an ObjectOutputStream wrapping the specified OutputStream
     *
     * @throws IOException
     */
    protected ObjectOutputStream createObjectOutputStream(OutputStream out)
            throws IOException
    {
        return new ObjectOutputStream(out);
    }
}
