package webwork.config.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import webwork.config.Configuration;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This class is a travesity of justice.  I feel dirty even writing this comment!
 * <p/>
 * However its part of the hopelessly inefficient Webwork1 giant registry lookup schmozzle that is {@link
 * webwork.config.Configuration} and {@link webwork.config.ConfigurationInterface}
 * <p/>
 * I have chosen to extend it to capture the attributes that are defined in the actions.xml.
 */
public class XMLConfigurationReader
{
    public static final Log log = LogFactory.getLog(XMLConfigurationReader.class);
    /**
     * This stores the mapping from URL -> action.  It looks through the (xml) configuration file and adds all the
     * commands, aliases, and views.
     * <p/>
     * One caveat - if you are using an extension aside from '.action', the views will be added with '.action' here, and
     * then replaced on retrieval.  This is to warn off a recursive dependency that I haven't had a chance to look at
     * yet.
     */
    private final Map<String, Object> actionMappings;
    private static final String EXTENSION_PROPERTY = "webwork.action.extension";
    private static final String DOT_ACTION_INFO_SUFFIX = ".actionInfo";
    private static final String DOT_ACTION_SUFFIX = ".action";
    private static final String DOT_ACTION_ROLES_SUFFIX = ".actionRoles";


    public XMLConfigurationReader(Element configurationRootElement)
    {
        this(configurationRootElement,null);
    }
    public XMLConfigurationReader(Element configurationRootElement, String sourceOfConfig)
    {
        actionMappings = getMappingsFromDocument(configurationRootElement, sourceOfConfig);
    }

    public Object getActionMapping(String mappingName)
    {
        final String key = replaceExtension(mappingName);
        return actionMappings.get(key);
    }

    public Set getActionMappingNames()
    {
        return actionMappings.keySet();
    }

    private Map<String, Object> getMappingsFromDocument(Element element, final String sourceOfConfig)
    {
        Map<String, Object> actionMap = new HashMap<String, Object>();

        // Get list of actions
        NodeList actions = element.getElementsByTagName("action");

        // Build list of views
        int length = actions.getLength();
        for (int i = 0; i < length; i++)
        {
            Element actionE = (Element) actions.item(i);
            String actionName = actionE.getAttribute("name");
            String actionAlias = actionE.getAttribute("alias");

            // build action alias to action name mapping
            buildActionMapping(actionMap, actionE, actionName, actionAlias, sourceOfConfig);


            // Commands
            buildCommandMappings(actionMap, actionE, actionName, actionAlias, sourceOfConfig);

            // Roles Required
            buildRolesRequiredMappings(actionMap, actionE, actionName, defaultActionAlias(actionAlias, actionName));
        }

        return actionMap;
    }

    private void buildRolesRequiredMappings(Map<String, Object> actionMap, Element actionElement, String actionName, String actionAlias)
    {
        // Root Actions
        if (log.isDebugEnabled())
        {
            log.debug(String.format("Adding roles required for action name: '%s' under alias: '%s", actionName, actionAlias));
        }
        actionMap.put(actionAlias + DOT_ACTION_ROLES_SUFFIX, rolesRequiredAttributeFor(actionElement));

        // Command Actions
        final NodeList commands = actionElement.getElementsByTagName("command");
        for (int j = 0; j < commands.getLength(); j++)
        {
            Element commandElement = (Element) commands.item(j);
            String commandName = commandElement.getAttribute("name");
            String commandAlias = commandElement.getAttribute("alias");

            if (!"".equals(commandAlias))
            {
                final String rolesRequiredForCommand = rolesRequiredAttributeFor(commandElement);
                if (!"".equals(rolesRequiredForCommand))
                {
                    actionMap.put(commandAlias + DOT_ACTION_ROLES_SUFFIX, rolesRequiredAttributeFor(commandElement));
                }
                else
                {
                    actionMap.put(commandAlias + DOT_ACTION_ROLES_SUFFIX, rolesRequiredAttributeFor(actionElement));
                }
            }
            else
            {
                final String rolesRequiredForCommand = rolesRequiredAttributeFor(commandElement);
                if (!"".equals(rolesRequiredForCommand))
                {
                    actionMap.put(actionAlias + "!" + commandName + DOT_ACTION_ROLES_SUFFIX, rolesRequiredAttributeFor(commandElement));
                }
                else
                {
                    actionMap.put(actionAlias + "!" + commandName + DOT_ACTION_ROLES_SUFFIX, rolesRequiredAttributeFor(actionElement));
                }
            }
        }
    }

    private String defaultActionAlias(String actionAlias, String actionName)
    {
        if ("".equals(actionAlias))
        {
            return actionName;
        }
        return actionAlias;
    }

    private String rolesRequiredAttributeFor(final Element actionE)
    {
        final List<Node> attributes = findAttributes(actionE);
        for (final Node attribute : attributes)
        {
            if (attribute.getNodeName().equals("roles-required"))
            {
                return attribute.getNodeValue();
            }
        }
        return "";
    }

    private void buildActionMapping(final Map<String, Object> actionMap, final Element actionE, final String actionName, final String actionAlias, final String sourceOfConfig)
    {
        if (!"".equals(actionAlias))
        {
            final String magicActionKey = actionAlias + DOT_ACTION_SUFFIX;
            log.debug("Adding action alias " + actionAlias + "=" + actionName);
            actionMap.put(magicActionKey, actionName);
        }

        ActionInfoImpl.Builder builder = ActionInfoImpl.builder(actionName, actionAlias)
                .setSource(sourceOfConfig)
                .addAttributes(allAttributes(actionE));

        // Build views for this action
        buildViewMappings(actionMap, actionE, actionName, actionAlias, builder);

        recordActionInfo(actionMap, actionName, actionAlias, builder);
    }

    private void buildViewMappings(final Map<String, Object> actionMap, final Element actionE, final String actionName, final String actionAlias, final ActionInfoImpl.Builder builder)
    {
        NodeList views = actionE.getElementsByTagName("view");
        for (int j = 0; j < views.getLength(); j++)
        {
            Element view = (Element) views.item(j);

            // This is to avoid listing "view" elements
            // that are associated with the commands
            // of this action
            if (!view.getParentNode().equals(actionE))
            {
                break;
            }

            // View mappings for this action
            NodeList viewMapping = view.getChildNodes();
            StringBuilder mapping = new StringBuilder();
            for (int k = 0; k < viewMapping.getLength(); k++)
            {
                Node mappingNode = viewMapping.item(k);
                if (mappingNode instanceof Text)
                {
                    mapping.append(mappingNode.getNodeValue());
                }
            }

            final String viewName = view.getAttribute("name");
            final String actionViewMapping = mapping.toString().trim();

            builder.startView(viewName, actionViewMapping)
                    .addAttributes(allAttributes(view))
                    .endView();

            String actionViewName;
            if ("".equals(actionAlias))
            {
                if (!"".equals(actionName))
                {
                    actionViewName = actionName + "." + viewName;
                }
                else
                {
                    actionViewName = viewName;
                }
            }
            else
            {
                actionViewName = actionAlias + "." + viewName;
            }
            log.debug("Adding view mapping " + actionViewName + "=" + actionViewMapping);
            actionMap.put(actionViewName, actionViewMapping);
        }
    }

    private void buildCommandMappings(final Map<String, Object> actionMap, final Element actionE, final String actionName, final String actionAlias, final String sourceOfConfig)
    {
        NodeList commands = actionE.getElementsByTagName("command");
        for (int j = 0; j < commands.getLength(); j++)
        {
            Element commandE = (Element) commands.item(j);
            String commandName = commandE.getAttribute("name");
            String commandAlias = commandE.getAttribute("alias");

            if (!commandAlias.equals(""))
            {
                final String magicCommandAliasKey = commandAlias + DOT_ACTION_SUFFIX;
                log.debug("Adding command alias " + commandAlias + "=" + actionName + "!" + commandName);
                actionMap.put(magicCommandAliasKey, actionName + "!" + commandName);
            }

            ActionInfoImpl.Builder builder = ActionInfoImpl.builder(commandName, commandAlias)
                    .setSource(sourceOfConfig)
                    .addAttributes(allAttributes(commandE));

            // Build views for this action
            NodeList views = commandE.getElementsByTagName("view");
            for (int k = 0; k < views.getLength(); k++)
            {
                Element view = (Element) views.item(k);

                // View mappings for this action
                NodeList viewMapping = view.getChildNodes();
                StringBuilder mapping = new StringBuilder();
                for (int l = 0; l < viewMapping.getLength(); l++)
                {
                    Node mappingNode = viewMapping.item(l);
                    if (mappingNode instanceof Text)
                    {
                        mapping.append(mappingNode.getNodeValue());
                    }
                }

                final String viewName = view.getAttribute("name");
                final String commandViewMapping = mapping.toString().trim();

                builder.startView(viewName, commandViewMapping)
                        .addAttributes(allAttributes(view))
                        .endView();

                String commandViewName;
                if (commandAlias.equals(""))
                {
                    if (actionAlias.equals(""))
                    {
                        commandViewName = actionName + "!" + commandName + "." + viewName;
                    }
                    else
                    {
                        commandViewName = actionAlias + "!" + commandName + "." + viewName;
                    }
                }
                else
                {
                    commandViewName = commandAlias + "." + viewName;
                }

                log.debug("Adding command view mapping " + commandViewName + "=" + commandViewMapping);
                actionMap.put(commandViewName, commandViewMapping);
            }

            recordActionInfo(actionMap, commandName, commandAlias, builder);

        }
    }

    private void recordActionInfo(final Map<String, Object> actionMap, final String actionName, final String actionAlias, final ActionInfoImpl.Builder builder)
    {
        ActionInfo actionInfo = builder.build();

        // it possible that we have no alias, in which case we want to use the name directly
        String actionPrefix = actionAlias;
        if ("".equals(actionAlias))
        {
            actionPrefix = actionName;
        }
        // put it in the map
        final String magicActionInfoKey = actionPrefix + DOT_ACTION_INFO_SUFFIX;
        actionMap.put(magicActionInfoKey, actionInfo);
    }


    private Map<String, String> allAttributes(final Element element)
    {
        final List<Node> attributes = findAttributes(element);
        final Map<String, String> attributeMap = new HashMap<String, String>();
        for (Node attribute : attributes)
        {
            attributeMap.put(attribute.getNodeName(), attribute.getNodeValue());
        }
        return attributeMap;
    }

    private List<Node> findAttributes(final Element actionE)
    {
        List<Node> attributes = new ArrayList<Node>();
        NamedNodeMap nodeList = actionE.getAttributes();
        for (int i = 0; i < nodeList.getLength(); i++)
        {
            Node child = nodeList.item(i);
            if (child.getNodeType() == Node.ATTRIBUTE_NODE)
            {
                attributes.add(child);
            }
        }
        return attributes;
    }

    /**
     * As the actions are stored in the action mapping with a '.action' extension, we need to map the current request to
     * that '.action' mapping.
     * <p/>
     * So an action 'ABC.jspa' would be lookup up in the map at 'ABC.action'.
     * <p/>
     * The extension used is retrieved from the configuration using key 'webwork.action.extension'
     *
     * @param actionName The original action (url) to be mapped, including an extension (if any).
     * @return The action name used in actionMappings - ie with a '.action' extension.
     */
    private String replaceExtension(String actionName)
    {
        //this prevents a possible recursion with getting this property
        if (EXTENSION_PROPERTY.equals(actionName))
        {
            return actionName;
        }

        String ext = "." + Configuration.get(EXTENSION_PROPERTY);

        //replace all custom extensions with .action to retrieve view mapping from cache
        if (actionName != null && !DOT_ACTION_SUFFIX.equals(ext))
        {
            // try to find another extension
            if (actionName.endsWith(ext))
            {
                actionName = actionName.substring(0, actionName.lastIndexOf(ext)) + DOT_ACTION_SUFFIX;
            }

            // otherwise look for something like .jspa? in the view mapping and replace
            // with .action?
            int idx = actionName.indexOf(ext + "?");

            if (idx > 0)
            {
                actionName = actionName.substring(0, idx) + ".action?" + actionName.substring(idx + ext.length() + 1);
            }
        }

        return actionName;
    }


}
