/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.examples.userreg;

import java.net.InetAddress;
import java.net.UnknownHostException;

//import javax.mail.Address;

/**
 * @author Kjetil H.Paulsen (kjetil@java.no)
 * @version $Revision: 1.3 $
 */

public class Validator
{

    /**
     * First multiplier used for social security number (norwegian)
     */
    private final static int[] SSN_MULTIPLIER =
            { 3, 7, 6, 1, 8, 9, 4, 5, 2, 1, 0 };

    /**
     * Second multiplier used for social security numbers (norwegian)
     */
    private final static int[] CONTROL_NO_MULTIPLIER =
            { 5, 4, 3, 2, 7, 6, 5, 4, 3, 2, 1 };

    /**
     * The number of digits in the number to be controlled
     */
    private final static int DIGIT_COUNT = 11;

    private Validator()
    {
    }

    public static boolean checkEmail(String email, boolean doNetCheck)
    {
        // do some checks, should have javax.mail available for this
        int index = email.indexOf('@');

        // the email have to contain the @ sign and there have to
        // be at least one character on the left side of it
        if ((index < 1) || ((index + 1) >= email.length()))
        {
            return false;
        }

        if (doNetCheck)
        {
            String host = email.substring(index + 1);
            try
            {
                InetAddress.getByName(host);
            }
            catch (UnknownHostException e)
            {
                return false;
            }
        }
        return true;

    }

    /**
     * @param ssn The social security number to check
     */
    public static boolean checkSsn(String ssn)
    {

        return ssn != null && ssn.trim().length() > 0; // haven't hade time to generate samples of control numbers you can use, so for now we return true
    }

    /**
     * This method checks if the SSN given is compliant with the gender given by the user. Females: the 9th number is a
     * odd number Males:   the 9th number is an even number
     *
     * @param ssn  The Norwegian social security number
     * @param male This is true if the user is a male
     */
    public static boolean checkSsnWithGenderCheck(String ssn, boolean male)
    {

        //if (isSSNMod11(ssn)) { // haven't hade time to generate samples of control numbers you can use, so for now we return true

        // check the 9th number for gender compliance
        int i = Integer.parseInt(ssn.substring(8, 9));
        i = i % 2;
        if ((i > 0) && male)
        {
            return true;
        }
        else if ((i == 0) && !male)
        {
            return true;
        }
        //}
        return false;

    }


    // not JavaDoc'et since not Webwork functionality
    /*
      * Control on the number using mod-11 arithmetic and the
      * specified multiplier
      *
      * @param	number number to be controlled
      *	@param	multiplier weight multiplier
      * @return	true if valid control, false otherwise
      */
    private static boolean isMod11(String number, int[] multiplier)
    {
        if (null == number || null == multiplier || multiplier.length != DIGIT_COUNT)
        {
            return false;
        }

        // Append leading zeros to the number
        while (number.length() < DIGIT_COUNT)
        {
            number = "0" + number;
        }

        int sum = 0;
        for (int i = 0; i <= DIGIT_COUNT - 1; ++i)
        {
            char ch = number.charAt(i);
            if (ch < '0' || '9' < ch)
            {
                return false;
            }

            sum += val(ch) * multiplier[i];
        }

        return (0 == (sum %= 11));
    }

    // not JavaDoc'et since not Webwork functionality
    /*
      * Controls whether a norwegian social security number is CDV valid
      * @param	ssn Social security number
      * @return	true if valid, false otherwise
      */
    private static boolean isSSNMod11(String ssn)
    {
        return
                isMod11(ssn, SSN_MULTIPLIER) &&
                        isMod11(ssn, CONTROL_NO_MULTIPLIER);
    }


    // not JavaDoc'et since not Webwork functionality
    /*
      * Get the numveric value of a character.
      * @param ch character between 0 and 9
      * @return numeric value of character
      */
    private final static int val(char ch)
    {
        if ('0' <= ch && ch <= '9')
        {
            //throw new Exception("Charachter must be numeric.");
        }
        return ch - '0';
    }


}
