package webwork.action.factory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.action.Action;
import webwork.config.Configuration;
import webwork.util.ClassLoaderUtils;
import webwork.util.classloader.WebworkClassLoader;
import webwork.util.injection.ObjectFactory;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.StringTokenizer;

public class ReloadingJavaActionFactory extends JavaActionFactory
{
    private static final Log log = LogFactory.getLog(ReloadingJavaActionFactory.class);
    //we store this here as well as the parent field to avoid having to cast to call methods on the classloader
    private WebworkClassLoader reloadingClassLoader;
    private URL source;

    public ReloadingJavaActionFactory()
    {
        URL url = getClass().getResource("/webwork.properties");
        if (url == null)
        {
            log.warn("Cannot find CodeSource for webwork, unable to creare reloading action factory");
        }
        else
        {
            try
            {
                String urlVal = url.toString();
                source = new URL(urlVal.substring(0, urlVal.lastIndexOf("webwork.properties")));
                log.debug("Creating classloader for url " + source);
                WebworkClassLoader actionLoader = WebworkClassLoader.getInstance(source, Thread.currentThread().getContextClassLoader());
                StringTokenizer tok = new StringTokenizer(Configuration.getString("webwork.action.packages").trim(), ", ");
                String[] packages = new String[tok.countTokens()];
                int counter = 0;
                while (tok.hasMoreTokens())
                {
                    packages[counter++] = tok.nextToken();
                }
                actionLoader.setPackages(packages);
                this.actionLoader = actionLoader;
                this.reloadingClassLoader = actionLoader;
            }
            catch (MalformedURLException ex)
            {
                log.error("Grr", ex);
            }
        }
    }

    /**
     * Returns a loaded and instantiated action class instance using a fully qualified classname.
     *
     * @param name classname of the action to be created
     *
     * @return the action corresponding to the given Java class name
     */
    public Action getActionImpl(String name) throws Exception
    {
        if (reloadingClassLoader != null && reloadingClassLoader.isStale())
        {
            synchronized (this)
            {
                log.debug("Classes modified, reloading...");
                reloadingClassLoader = (WebworkClassLoader) reloadingClassLoader.clone();
                actionLoader = reloadingClassLoader;
            }
        }
        Class actionClass = null;
        try
        {
            actionClass = reloadingClassLoader.loadClass(name);
        }
        catch (Exception e)
        {
            try
            {
                actionClass = ClassLoaderUtils.loadClass(name, getClass());
            }
            catch (Exception e2)
            {
                // Not found
                throw new IllegalArgumentException("Action '" + name + "' not found");
            }
        }
        try
        {
            Action action = (Action) ObjectFactory.instantiate(actionClass);
            return action;
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("Action '" + name + "' could not be instantiated:" + e);
        }
    }
}
