/**
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.xslt.SAXAdapter;

import org.apache.commons.logging.*;
import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;


/**
 * XMLReaderAdapter converts a java bean to its xml representation. It does this by simulating a SAX XMLReader which an
 * xslt processor can use as a SAXSource.
 * <p/>
 * The actual work is done in XMLWalker as the interface XMLReader is stateful in respect to the input source. So this
 * class is actually a facade to the stateless class @see XMLWalker
 *
 * @author <a href="mailto:meier@o-matic.de">Philipp Meier</a>
 */
public class XMLReaderAdapter implements Locator, XMLReader
{

    private String systemId;
    private Object object;
    private String root = null;
    private ContentHandler contentHandler = new DefaultHandler();
    private XMLWalker walker;

    public XMLReaderAdapter(XMLWalker walker, Object object)
    {
        this(walker, object, "data");
    }

    public XMLReaderAdapter(XMLWalker walker, Object object, String root)
    {
        this.walker = walker;
        this.object = object;
        this.root = root;
    }

    public void parse() throws IOException, SAXException
    {
        if (object == null)
        {
            throw new SAXException("no object to parse defined " + "(use XMLReaderAdapter.setObject)");
        }
        if (contentHandler == null)
        {
            throw new SAXException("no content handler defined " + "(use XMLReaderAdapter.setContentHandler");
        }
        contentHandler.setDocumentLocator(this);
        contentHandler.startDocument();
        /** Keeps track of walked instances to detect circular references. **/
        List walkedInstances = new ArrayList();
        walker.walk(contentHandler, object, root, walkedInstances);
        contentHandler.endDocument();
    }

    /**
     * SAX Reader/Locator Dummy Implementation
     */
    public String getPublicId()
    {
        return null;
    }

    public String getSystemId()
    {
        return systemId;
    }

    public int getLineNumber()
    {
        return -1;
    }

    public int getColumnNumber()
    {
        return -1;
    }

    public void setEntityResolver(EntityResolver resolver)
    {
    }

    public EntityResolver getEntityResolver()
    {
        return null;
    }

    public void setDTDHandler(DTDHandler handler)
    {
    }

    public DTDHandler getDTDHandler()
    {
        return null;
    }

    public void setContentHandler(ContentHandler handler)
    {
        this.contentHandler = handler;
    }

    public ContentHandler getContentHandler()
    {
        return contentHandler;
    }

    public void setErrorHandler(ErrorHandler handler)
    {
    }

    public ErrorHandler getErrorHandler()
    {
        return null;
    }

    public void parse(InputSource source) throws IOException, SAXException
    {
        parse();
    }

    public void parse(String s) throws IOException, SAXException
    {
        parse();
    }

    public void setObject(Object object)
    {
        this.object = object;
    }

    public void setRoot(String root)
    {
        this.root = root;
    }

    public boolean getFeature(String featureId) throws SAXNotRecognizedException
    {
        return false;
    }

    public void setFeature(String featureId, boolean on) throws SAXNotRecognizedException, SAXNotSupportedException
    {
    }

    public Object getProperty(String name) throws SAXNotRecognizedException
    {
        throw new SAXNotRecognizedException(name);
    }

    public void setProperty(String propertyId, Object property)
            throws SAXNotRecognizedException, SAXNotSupportedException
    {
    }

}
