/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.taglib;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.util.BeanUtil;
import webwork.util.TextUtil;

import java.util.Iterator;
import java.io.IOException;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.PageContext;

/**
 * Access the value of a named property. By default (implicitly), this tag will escape its contents if it does *not*
 * have a body. If it does have a body, the tag will not escape the contents. You can explicitly tell the tag to escape
 * or not. Quoted text that is escaped will have its quotes stripped off.
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @author Matt Baldree (matt@smallleap.com)
 * @version $Revision: 1.10 $
 */
public class BasicPropertyTag
        extends WebWorkBodyTagSupport
{
    protected static Log log = LogFactory.getLog(BasicPropertyTag.class);

    // Attributes ----------------------------------------------------
    protected String valueAttr;
    protected Boolean escape;
    protected boolean hadBody;

    // Public --------------------------------------------------------
    public void setValue(String inName)
    {
        valueAttr = inName;
    }

    public void setEscape(boolean inEscape)
    {
        escape = inEscape;
    }

    // BodyTag implementation ----------------------------------------
    public int doStartTag() throws JspException
    {
        hadBody = false;
        Object value = findValue(valueAttr);
        getStack().pushValue(value);

        String id = getId();
        if (id != null && value != null)
        {
            pageContext.setAttribute(id, value);
            pageContext.setAttribute(id, value, PageContext.REQUEST_SCOPE);
        }
        return EVAL_BODY_BUFFERED;
    }

    public int doAfterBody() throws JspException
    {
        // This is a workaround for a bug in Jasper.
        // http://znutar.cortexity.com/BugRatViewer/ShowReport/652
        // Resin 1.2.1 also has this bug (i.e. body is called even though it is empty)

        // If the bodyContent is null then just return
        if (this.bodyContent == null)
        {
            return SKIP_BODY;
        }
        String bodyContent = null;
        try
        {
            bodyContent = this.bodyContent.getString();
        }
        catch (Exception e)
        {
            //do nothing - leave body as null
        }

        //Added null check - Fixes WW-195 on Resin.
        hadBody = bodyContent != null && bodyContent.length() != 0;

        if (hadBody)
        {
            try
            {
                if (shouldEscape())
                {
                    bodyContent = escapeValue(bodyContent);
                }
                onOutput(this.bodyContent.getEnclosingWriter(), bodyContent);
            }
            catch (java.io.IOException e)
            {
                throw new JspException("Could not show attribute " + valueAttr, e);
            }
        }
        return SKIP_BODY;
    }

    public int doEndTag() throws JspException
    {
        Object value = getStack().popValue();

        if (!hadBody)
        {
            // No body used. This means that we want to print the parameter out
            try
            {
                String bodyOutput = "";
                if (value != null && getId() == null)
                {
                    if (value instanceof Iterator)
                    {
                        Iterator iter = (Iterator) value;
                        if (iter.hasNext())
                        {
                            if (shouldEscape())
                            {
                                bodyOutput = escapeValue(BeanUtil.toStringValue(iter.next()));
                            }
                            else
                            {
                                bodyOutput = BeanUtil.toStringValue(iter.next());
                            }
                        }
                    }
                    else if (value instanceof char[])
                    {
                        if (shouldEscape())
                        {
                            bodyOutput = escapeValue(String.valueOf((char[]) value));
                        }
                        else
                        {
                            bodyOutput = String.valueOf((char[]) value);
                        }
                    }
                    else
                    {
                        if (shouldEscape())
                        {
                            bodyOutput = escapeValue(BeanUtil.toStringValue(value));
                        }
                        else
                        {
                            bodyOutput = BeanUtil.toStringValue(value);
                        }
                    }
                }
                onOutput(pageContext.getOut(), bodyOutput);
            }
            catch (Throwable t)
            {
                String msg = "Could not show value: " + valueAttr;
                throw new JspException(msg + ", throwable: " + t, t);
            }
        }

        super.doEndTag();
        return EVAL_PAGE;
    }

    /**
     * This is called when the tag has something to output.  By default it does just that, it writes the output
     * but it you may override this method and do something else if you like
     * 
     * @param out the JspWriter to use
     * @param bodyOutput the content to write
     * @throws IOException because we are involved in that game
     */
    protected void onOutput(JspWriter out, String bodyOutput) throws IOException
    {
        out.write(bodyOutput);
    }

    /**
     * The BasicPropertyTag should escape IF the escape value has been set to true. At all other times it will false (ie
     * not escape data).
     * <p/>
     * CHANGE THIS METHOD ON PAIN OF DEATH -mike :)
     *
     * @see PropertyTag#shouldEscape()
     */
    protected boolean shouldEscape()
    {
        if (escape == null)
        {
            return false;
        }

        return escape.booleanValue();
    }

    /**
     * This is called to escape a value for output.  By default it uses HTML escaping.  A subclass can overrride this
     * and use some other escaping mechanism
     *
     * @param value the value to be escaped
     *
     * @return the escaped value
     */
    protected String escapeValue(String value)
    {
        return TextUtil.escapeHTML(value);
    }
}


