package webwork.interceptor;

import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;

import webwork.action.Action;

/**
 * TODO: Document this class / interface here
 *
 * @since v4.2
 */
public class ChainedInterceptorFactory implements InterceptorFactory
{
    private final ConcurrentMap<String, InterceptorFactoryHolder> factoryRegistrar =
            new ConcurrentHashMap<String, InterceptorFactoryHolder>();
    private final SortedSet<InterceptorFactoryHolder> factories =
            new ConcurrentSkipListSet<InterceptorFactoryHolder>(new InterceptorFactoryHolderComparator());

    private static final ChainedInterceptorFactory instance = new ChainedInterceptorFactory();

    public static ChainedInterceptorFactory getInstance()
    {
        return instance;
    }

    public Interceptor createInterceptorFor(final Action action)
    {
        return new ChainedInterceptor(createInterceptorsFor(action));
    }

    private List<Interceptor> createInterceptorsFor(Action action)
    {
        List<Interceptor> interceptors = new LinkedList<Interceptor>();
        for (InterceptorFactoryHolder holder : factories)
        {
            Interceptor interceptor = holder.factory.createInterceptorFor(action);
            if (interceptor != null)
            {
                interceptors.add(interceptor);
            }
        }
        return interceptors;
    }

    public synchronized void register(String key, int weight, InterceptorFactory factory)
    {
        if (key == null) throw new IllegalArgumentException("Key is null");
        if (factory == null) throw new IllegalArgumentException("Factory is null.");
        InterceptorFactoryHolder holder = new InterceptorFactoryHolder(key, factory, weight);
        InterceptorFactoryHolder replaced = factoryRegistrar.put(key, holder);
        if (replaced != null)
        {
            factories.remove(replaced);
        }
        factories.add(holder);
    }

    public synchronized boolean unregister(String key, InterceptorFactory factory)
    {
        if (key == null) throw new IllegalArgumentException("Key is null");
        if (factory == null) throw new IllegalArgumentException("Factory is null.");
        InterceptorFactoryHolder holder = factoryRegistrar.get(key);
        if (holder == null)
        {
            return false;
        }
        if (holder.factory != factory)
        {
            return false;
        }
        factories.remove(holder);
        return factoryRegistrar.remove(key, holder);
    }

    private static class InterceptorFactoryHolder
    {
        private final InterceptorFactory factory;
        private final String key;
        private final int weight;

        InterceptorFactoryHolder(String key, InterceptorFactory factory, int weight)
        {
            this.key = key;
            this.factory = factory;
            this.weight = weight;
        }

        @Override
        public boolean equals(Object o)
        {
            if (this == o)
            {
                return true;
            }
            if (o == null || getClass() != o.getClass())
            {
                return false;
            }

            InterceptorFactoryHolder that = (InterceptorFactoryHolder) o;

            if (factory != that.factory)
            {
                return false;
            }
            if (key != null ? !key.equals(that.key) : that.key != null)
            {
                return false;
            }

            return true;
        }

        @Override
        public int hashCode()
        {
            return key.hashCode();
        }
    }

    private static class InterceptorFactoryHolderComparator implements Comparator<InterceptorFactoryHolder>
    {
        public int compare(InterceptorFactoryHolder lhs, InterceptorFactoryHolder rhs)
        {
            int diff = lhs.weight - rhs.weight;
            if (diff != 0)
            {
                return diff;
            }
            if (lhs.equals(rhs))
            {
                return 0;
            }
            return lhs.key.compareTo(rhs.key);
        }
    }
}
