/*
* WebWork, Web Application Framework
*
* Distributable under Apache license.
* See terms of license at opensource.org
*/
package webwork.view.velocity;

import webwork.util.URLCodec;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpUtils;

/**
 * A utility class that is used to URL-encode links in Velocity templates.
 *
 * @author Dave Bryson (daveb@miceda-data.com)
 */
public class URLBean
{
    /**
     * Http Request
     */
    private HttpServletRequest request = null;
    /**
     * Http Response
     */
    private HttpServletResponse response = null;
    /**
     * Content name
     */
    private String page = null;
    /**
     * Path info parameters
     */
    private Map params = new HashMap();

    public URLBean()
    {

    }

    /**
     * @return String the page name
     */
    public String getPage()
    {
        return page;
    }

    /**
     * Set the page.
     *
     * @param page the page name
     *
     * @return URLBean this
     */
    public URLBean setPage(String page)
    {
        this.page = page;
        return this;
    }

    /**
     * Add a parameter to the URL. Cut and paste from the UrlTag in WebWork
     *
     * @param name  the parameter name
     * @param value the parameter value
     *
     * @return URLBean this
     */
    public URLBean addParameter(String name, Object value)
    {
        if (value == null)
        {
            params.remove(name);
        }
        else
        {
            params.put(name, value.toString());
        }

        return this;
    }

    /**
     * Set the HttpRequest
     *
     * @param req the request
     */
    public void setRequest(HttpServletRequest req)
    {
        this.request = req;

        // Add query parameters
        String query = req.getQueryString();
        if (query != null)
        {
            // Remove possible #foobar suffix
            int idx = query.lastIndexOf('#');
            if (idx != -1)
            {
                query = query.substring(0, idx - 1);
            }

            Hashtable queryParams = HttpUtils.parseQueryString(query);
            if (queryParams.size() > 0)
            {
                if (params == null)
                {
                    params = new HashMap();
                }
                params.putAll(queryParams);
            }
        }
    }

    /**
     * Set the HttpServletResponse
     *
     * @param resp the response
     */
    public void setResponse(HttpServletResponse resp)
    {
        this.response = resp;
    }

    /**
     * Generates the URL. This is called automatically in the Velocity template. In otherwords, you do not need to
     * explicitly call toString() This is pretty much a cut and paste of code from the URLTag in Webwork
     *
     * @return String the generated URI
     */
    public String getURL()
    {
        StringBuilder link = new StringBuilder();
        if (page != null)
        {
            if (page.startsWith("/"))
            {
                // Add context to the front of the URI
                String appContext = request.getContextPath();
                link.append(appContext);
            }

            link.append(page);
        }
        else
        {
            // Go to "same page"
            String requestURI = (String) request.getAttribute("webwork.request_uri");

            if (requestURI == null)
            {
                requestURI = request.getRequestURI();
            }
            link.append(requestURI);
        }

        if (params.size() > 0)
        {
            link.append('?');

            // Set parameters
            Iterator e = params.entrySet().iterator();
            while (e.hasNext())
            {
                Map.Entry entry = (Map.Entry) e.next();
                String name = (String) entry.getKey();
                Object value = entry.getValue();
                if (value != null)
                {
                    if (value instanceof String)
                    {
                        link.append(name);
                        link.append('=');
                        link.append(URLCodec.encode((String) value));
                    }
                    else
                    {
                        String[] values = (String[]) value;

                        for (int i = 0; i < values.length; i++)
                        {
                            String val = values[i];
                            link.append(name);
                            link.append('=');
                            link.append(URLCodec.encode(val));
                            if (i < values.length - 1)
                            {
                                link.append("&amp;");
                            }
                        }

                    }
                }

                if (e.hasNext())
                {
                    link.append("&amp;");
                }
            }
        }

        String result;
        try
        {
            result = response.encodeURL(link.toString());
        }
        catch (Exception e)
        {
            // Could not encode URL for some reason
            // Use it unchanged
            result = link.toString();
        }

        return result;
    }

    public String toString()
    {
        return getURL();
    }
} 
