/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.taglib;

import org.apache.commons.logging.*;
import webwork.util.MakeIterator;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspTagException;
import javax.servlet.jsp.PageContext;
import java.util.*;
import java.lang.reflect.Array;

/**
 * Tag to iterate over an iterable value.
 * <p/>
 * An iterable value can be either of: java.util.Collection, java.util.Iterator, java.util.Enumeration, java.util.Map,
 * array, XML Node, or XML NodeList
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @version $Revision: 1.9 $
 */
public class IteratorTag
        extends WebWorkBodyTagSupport
{
    // Attributes ----------------------------------------------------
    protected String valueAttr;
    protected String statusAttr;
    protected Object currentValue;
    protected Iterator iterator;
    protected IteratorStatus status;
    protected IteratorStatus.StatusState statusState;
    protected String statusName;
    protected Object oldStatus;

    // Public --------------------------------------------------------

    /**
     * Set the name of the property whose value is iterable.
     *
     * @param aName
     */
    public void setValue(String aName)
    {
        valueAttr = aName;
    }

    /**
     * Set the name of the status object that is to be exposed.
     *
     * @param aName
     */
    public void setStatus(String aName)
    {
        statusAttr = aName;
    }

    // BodyTag implementation ----------------------------------------
    public int doStartTag()
    {
        if (statusAttr != null)
        {
            statusState = new IteratorStatus.StatusState();
            status = new IteratorStatus(statusState);
        }

        Object value = findValue(valueAttr);

        if (value != null)
        {
            // Make sure it is iterable
            iterator = MakeIterator.convert(value);

            // Get first item
            if (iterator.hasNext())
            {
                currentValue = iterator.next();
                getStack().pushValue(currentValue);

                String id = getId();
                if (id != null && currentValue != null)
                {
                    pageContext.setAttribute(id, currentValue);
                    pageContext.setAttribute(id, currentValue, PageContext.REQUEST_SCOPE);
                }

                // Status object
                if (statusAttr != null)
                {
                    statusState.setLast(!iterator.hasNext());
                    statusName = (String) findValue(statusAttr);
                    oldStatus = pageContext.getAttribute(statusName);
                    pageContext.setAttribute(statusName, status);
                }

                return EVAL_BODY_TAG;
            }
            else
            {
                return SKIP_BODY;
            }
        }
        else
        {
            LogFactory.getLog(IteratorTag.class.getName()).warn("Value is null! Returning an empty set.");
            return SKIP_BODY;
        }
    }

    public int doAfterBody()
            throws JspException
    {
        getStack().popValue();

        if (iterator.hasNext())
        {
            currentValue = iterator.next();

            // Update status
            if (status != null)
            {
                statusState.next(); // Increase counter
                statusState.setLast(!iterator.hasNext());
            }

            getStack().pushValue(currentValue);
            String id = getId();
            if (id != null && currentValue != null)
            {
                pageContext.setAttribute(getId(), currentValue);
                pageContext.setAttribute(getId(), currentValue, PageContext.REQUEST_SCOPE);
            }
            return EVAL_BODY_TAG;
        }
        else
        {
            // Reset status object (if someone else has set one with same name)
            if (status != null)
            {
                if (oldStatus == null)
                {
                    pageContext.removeAttribute(statusName);
                }
                else
                {
                    pageContext.setAttribute(statusName, oldStatus);
                }
            }

            // Release objects
            currentValue = null;
            iterator = null;
            status = null;

            // Write result
            try
            {
                bodyContent.writeOut(bodyContent.getEnclosingWriter());
            }
            catch (Exception e)
            {
                throw new JspException(e.getMessage());
            }

            return SKIP_BODY;
        }
    }
}


