/*
* WebWork, Web Application Framework
*
* Distributable under Apache license.
* See terms of license at opensource.org
*/
package webwork.util;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.servlet.jsp.PageContext;

/**
 * Value stack implementation for a servlet based scenario.
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @author Maurice C. Parker (maurice@vineyardenterprise.com)
 * @version $Revision: 1 $
 */
public class ServletValueStack extends ValueStack
{
    // Constructor ---------------------------------------------------
    public ServletValueStack()
    {
        super();
    }

    /**
     * Get the value stack for a given request. If there is no stack available, create one.
     *
     * @param request the request for which a stack shall be returned
     *
     * @return the value stack
     */
    public static ServletValueStack getStack(ServletRequest request)
    {
        ServletValueStack stack = (ServletValueStack) request.getAttribute(STACK_NAME);
        if (stack == null)
        {
            stack = new ServletValueStack();
            request.setAttribute(STACK_NAME, stack);
        }
        stack.setRequest(request);
        return stack;
    }

    /**
     * Get the value stack for a given page context. If there is no stack available, create one.
     *
     * @param context the page context to associate with the stack
     *
     * @return the value stack
     */
    public static ServletValueStack getStack(PageContext context)
    {
        ServletRequest request = context.getRequest();
        ServletValueStack stack = (ServletValueStack) request.getAttribute(STACK_NAME);
        if (stack == null)
        {
            stack = new ServletValueStack();
            request.setAttribute(STACK_NAME, stack);
        }
        stack.setRequest(request);
        stack.setContext(context);
        return stack;
    }

    private ServletRequest request;
    private PageContext context;

    // Private -------------------------------------------------------
    /**
     * Set the request that should be associated with this stack. This is called for each use of the stack so that
     * attributes can be accessed properly.
     */
    private void setRequest(ServletRequest request)
    {
        this.request = request;
    }

    /**
     * Set the page context that should be associated with this stack. This is called for each use of the stack in a JSP
     * context.
     */
    public void setContext(PageContext context)
    {
        this.context = context;
    }

    protected Object findInContext(String id)
    {
        if (context != null)
        {
            try
            {
                return context.findAttribute(id); // Used in a JSP environment
            }
            catch (Exception e)
            {
                // This exception occurred alot before when the page context was not
                // updated correctly in the stack. I do not know if the comment and
                // code below is correct and needed anymore, but we keep it for safety

                //needed for apparent bug in Resin 2.1.1 :(
                return findAttribute(id);
            }
        }
        else
        {
            // If the findInContext method is called outside of a jsp page then there
            // is no page context and this code will execute.
            return findAttribute(id);
        }
    }

    /**
     * Mimic the behaviour of the findAttribute method in the PageContext This method is used when there is no
     * pageContext set for this stack and the findInContext method is called. Look for the attribute in the request,
     * session and application scope
     *
     * @return found value or null
     */
    protected Object findAttribute(String id)
    {
        Object result;

        // First look in the request
        if ((result = request.getAttribute(id)) != null)
        {
            return result;
        }
        // If nothing found then look in the session if there is one
        if (!(request instanceof HttpServletRequest))
        {
            return null;
        }
        HttpSession session = ((HttpServletRequest) request).getSession(false);
        if (session != null)
        {
            if ((result = session.getAttribute(id)) != null)
            {
                return result;
            }
            // Finally look in the application scope
            return session.getServletContext().getAttribute(id);
        }
        // We should really look in the application scope here as well,
        // but we do not have any way of finding it now
        return null;
    }

    /**
     * Get parameter directly from request
     */
    protected Object getParameter(String aName)
    {
        return request.getParameter(aName);
    }

}
