package webwork.util;

import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.LogConfigurationException;
import webwork.config.Configuration;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * This class allows for basic 'AOP' style around processing of method invocations.  The most common usage would be to
 * profile or log method calls.
 * <p/>
 * To use your own class, you need to implement {@link InjectionImpl}, and set the webwork property
 * <code>webwork.injection.class</code> to the class that you wish to use.
 *
 * @author <a href="mailto:scott@atlassian.com">Scott Farquhar</a>
 */
public class InjectionUtils
{
    private static InjectionImpl impl;
    public static final String CLASS_NAME = "webwork.injection.class";

    static
    {
        impl = new DefaultInjectionImpl();

        try
        {
            String classname = (String) Configuration.get(CLASS_NAME);
            if (classname != null && classname.length() > 0)
            {
                try
                {
                    impl = (InjectionImpl) ClassLoaderUtils.loadClass(classname, InjectionUtils.class).newInstance();
                }
                catch (Exception e)
                {
                    LogFactory.getLog(InjectionUtils.class).error("Could not load class " + classname + " or could not cast it to InjectionUtils.  Using default", e);
                }
            }
        }
        catch (IllegalArgumentException e)
        {
            //do nothing - this is just because the property couldn't be found.
        }
        catch (LogConfigurationException e)
        {
            //just catching this to be safe.
        }
    }

    public static Object invoke(Method target, Object value, Object[] args)
            throws IllegalAccessException, InvocationTargetException
    {
        return impl.invoke(target, value, args);
    }

    public static interface InjectionImpl
    {
        public Object invoke(Method target, Object value, Object[] args)
                throws IllegalAccessException, InvocationTargetException;
    }

    public static class DefaultInjectionImpl implements InjectionImpl
    {
        public Object invoke(Method target, Object value, Object[] args)
                throws IllegalAccessException, InvocationTargetException
        {
            return target.invoke(value, args);
        }
    }

}
