/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */

package webwork.view.xslt.SAXAdapter;

import org.apache.commons.logging.*;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import webwork.view.xslt.SAXAdapter.dom.NodeWalker;
import webwork.view.xslt.SAXAdapter.dom.TextWalker;

import java.util.*;

/**
 * This class is used by XMLReaderAdapter to do the actual walking of the object tree. Because the XMLReader Interface
 * is stateful in respect of the systemId (here the walked bean) this class implements the stateless part.
 */
public class XMLWalker
{
    // Walkers that are used without been registered due to
    // the nature of the classes they walk.
    private Walker arrayWalker = new ArrayWalker();
    private Walker beanWalker = new BeanWalker();
    private Walker toStringWalker = new ToStringWalker();
    private Walker nullWalker = new NullWalker();

    // walker registry
    private List walkers = new ArrayList();
    private Map walkerCache = new HashMap();

    public XMLWalker()
    {

        // Add default walkers
        registerWalker(new CollectionWalker());
        registerWalker(new EnumerationWalker());
        registerWalker(new MapWalker());
        registerWalker(new BooleanWalker());
        registerWalker(new CharacterWalker());

        // Add DOM walkers
        registerWalker(new TextWalker());
        registerWalker(new NodeWalker());
        // Arrays and JavaBeans are being processed specially using Class.isArray() but register it
        // here for completeness.
        registerWalker(arrayWalker);
        registerWalker(beanWalker);
        registerWalker(toStringWalker);
    }

    public void walk(ContentHandler contentHandler, Object object, String name, List walkedInstances)
            throws SAXException
    {
        //Null property handling:
        if (object == null)
        {
            nullWalker.walk(this, contentHandler, object, name, walkedInstances);
            return; // fast exit.
        }

        //Extract the object's class.
        Class klass = object.getClass();

        //Try to find a registered walker
        Walker walker = findWalker(klass);

        if (walker != null)
        {
            //Found special walker
            walker.walk(this, contentHandler, object, name, walkedInstances);
        }
        else if (klass.isArray())
        {
            //Use arrayWalker to process the object
            arrayWalker.walk(this, contentHandler, object, name, walkedInstances);
        }
        else if (String.class.isAssignableFrom(klass) || klass.isPrimitive() || Number.class.isAssignableFrom(klass))
        {
            //Primitives and java.lang.Numbers are handled by toStringWalker, too.
            toStringWalker.walk(this, contentHandler, object, name, walkedInstances);
        }
        else
        {
            //Fallback: dissect the object using the JavaBeans API.
            beanWalker.walk(this, contentHandler, object, name, walkedInstances);
        }
    }

    /**
     * Registers locally the class the walker returns as it's walkable type.
     */
    public void registerWalker(Walker walker)
    {
        //Registering a new walker will render the cached information obsolete.
        walkerCache.clear();
        this.walkers.add(walker);
    }

    /**
     * Find a walker registered for object's class. To speed up the walking of the collection of registered Walkers, a
     * cache is used. If no Walker was found null is returned.
     *
     * @return the Walker which was registered for the object's class or null if no walker was registered.
     */
    private synchronized Walker findWalker(Class klass)
    {

        // Try to look up walker in cache.
        if (walkerCache.containsKey(klass))
        {
            return (Walker) walkerCache.get(klass);
        }

        // Try to find walker in registred walkers.
        Walker walker = null;
        findWalker:
        for (Iterator i = walkers.iterator(); i.hasNext();)
        {
            Walker tryWalker = (Walker) i.next();
            if (tryWalker.getWalkedType().isAssignableFrom(klass))
            {
                walker = tryWalker;
                break findWalker;
            }
        }

        LogFactory.getLog(this.getClass()).debug("Caching " + (walker != null ? "walker " + walker.getClass().getName() :
                "no walker") + " for " + klass.getName());
        walkerCache.put(klass, walker);
        return walker;
    }

    /**
     * Expose toStringWalker to public for BeanWalker can use it as a fall back when to properties are found.
     */
    public Walker getToStringWalker()
    {
        return toStringWalker;
    }
}
