package webwork.util.classloader;

import java.security.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.io.*;
import java.util.*;

/**
 * @author Hani Suleiman (hani@formicary.net) Date: Oct 19 2003 Time: 11:16:12 PM
 */
public abstract class WebworkClassLoader extends SecureClassLoader implements Cloneable
{
    protected String[] packages = null;
    private Map cache = new HashMap();

    protected WebworkClassLoader()
    {
        this(ClassLoader.getSystemClassLoader());
    }

    protected WebworkClassLoader(ClassLoader parent)
    {
        super(parent);
    }

    public void setPackages(String[] p)
    {
        this.packages = p;
    }

    protected synchronized Class loadClass(String name, boolean resolve) throws ClassNotFoundException
    {
        Class c = (Class) cache.get(name);
        if (c != null)
        {
            return c;
        }
        boolean handles = false;
        if (packages != null)
        {
            for (int i = 0; i < packages.length; i++)
            {
                if (name.startsWith(packages[i]))
                {
                    handles = true;
                    break;
                }
            }
        }
        if (!handles)
        {
            return super.loadClass(name, resolve);
        }
        try
        {
            c = findClass(name);
        }
        catch (ClassNotFoundException ex)
        {
            return super.loadClass(name, resolve);
        }
        cache.put(name, c);
        return c;
    }

    protected Class findClass(String name) throws ClassNotFoundException
    {
        String path = name.replace('.', '/').concat(".class");
        byte[] data = getFile(path);
        if (data == null)
        {
            throw new ClassNotFoundException();
        }

        return defineClass(name, data, 0, data.length);
    }

    protected URL findResource(String name)
    {
        byte[] data = this.getFile(name);

        if (data == null)
        {
            return null;
        }
        try
        {
            return getDataURL(name, data);
        }
        catch (MalformedURLException e)
        {
            return null;
        }
    }

    protected abstract URL getDataURL(String name, byte[] data) throws MalformedURLException;

    protected Enumeration findResources(String name)
    {
        URL url = this.findResource(name);

        if (url == null)
        {
            return null;
        }

        return Collections.enumeration(Collections.singleton(url));
    }

    public abstract boolean isStale();

    protected abstract byte[] getFile(String path);

    public static WebworkClassLoader getInstance(URL url)
    {
        return getInstance(url, ClassLoader.getSystemClassLoader());
    }

    public static WebworkClassLoader getInstance(URL url, ClassLoader parent)
    {
        WebworkClassLoader loader;
        File file = new File(url.getFile());
        if (file.isDirectory())
        {
            loader = new DirectoryClassLoader(file, parent);
        }
        else
        {
            loader = new JarClassLoader(file, parent);
        }
        return loader;
    }

    public abstract Object clone();
}
