/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.util;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * ValueStack Query. This class parsers the ValueStack Query string and caches the parsed query.
 *
 * @author Maurice C. Parker (maurice@vineyardenterprise.com)
 * @version $Revision: 1.25 $
 */
public class Query
{
    // Static  -------------------------------------------------------
    protected static ConcurrentMap<String,Query> queriesMap = new ConcurrentHashMap<String,Query>();
    public static final Query CURRENT = getQuery(".");

    /**
     * Create a new Query and cache it for faster processing;
     */
    public static Query getQuery(String queryString)
    {
        //LogFactory.getLog(this.getClass()).debug( "GET_QUERY: " + queryString );
        Query query = queriesMap.get(queryString);

        if (query == null)
        {
            query = new Query(queryString);
            queriesMap.put(queryString, query);
        }

        return query;
    }

    // Attributes ----------------------------------------------------
    private final static char LPAREN = '(';
    private final static char RPAREN = ')';
    private final static char RBRACE = '}';
    private final static char LBRACE = '{';
    private final static char LBRACKET = '[';
    private final static char RBRACKET = ']';
    private final static char PARAM = '$';
    private final static char ATTR = '@';
    private final static char SLASH = '/';
    private final static char SQUOTE = '\'';
    private final static char PERIOD = '.';
    private final static char COMMA = ',';
    private final static char CONCAT = '+';

    private QuerySegment[] segments = new QuerySegment[5];
    private int segmentsIdx = 0;
    private String queryString;

    // Constructor ---------------------------------------------------
    private Query(String queryString)
    {
        // Trim the query of any surrounding whitespace
        this.queryString = queryString.trim();
        char[] query = this.queryString.toCharArray();

        // Check if we have an empty string ... can happen if the CONCAT character (+) occurs twice
        if (query.length == 0)
        {
            add(new QuerySegment(QuerySegment.NULL));
            return;
        }

        // Current value on top of stack
        if (query[0] == PERIOD && query.length == 1)
        {
            add(new QuerySegment(QuerySegment.CURRENT));
            return;
        }

        // Root value of stack
        if (query[0] == SLASH && query.length == 1)
        {
            add(new QuerySegment(QuerySegment.ROOT));
            return;
        }

        // The true keyword

        if (query.length == 4 &&

                query[0] == 't' && query[1] == 'r' &&

                query[2] == 'u' && query[3] == 'e')
        {

            add(new QuerySegment(QuerySegment.TRUE));

            return;

        }


        // The false keyword

        if (query.length == 5 &&

                query[0] == 'f' && query[1] == 'a' &&

                query[2] == 'l' && query[3] == 's' &&

                query[4] == 'e')
        {

            add(new QuerySegment(QuerySegment.FALSE));

            return;
        }

        // The null keyword

        if (query.length == 4 &&
                query[0] == 'n' && query[1] == 'u' &&

                query[2] == 'l' && query[3] == 'l')
        {

            add(new QuerySegment(QuerySegment.NULL));

            return;
        }
        // Look for concatenation expressions

        // First try to split the query based on the concat character

        List concatList = splitQuery(query, CONCAT);

        // If the concatList contains items then this is a concat query

        if (concatList != null)
        {

            QuerySegment qs = new QuerySegment(QuerySegment.CONCAT);

            for (int i = 0; i < concatList.size(); i++)
            {

                // Look up the Query for the string and add it to the segment

                Query concatQuery = getQuery((String) concatList.get(i));

                qs.addValue(concatQuery);

            }
            add(qs);

            return;

        }


        // Parameters

        if (query[0] == PARAM)
        {

            String id = new String(query, 1, query.length - 1);

            add(new QuerySegment(id, QuerySegment.PARAMETER));

            return;
        }

        // Integers (first check to see if the first character is a number)
        // If it is a number starting with . then there must be at least
        // one more character and it must be a number
        if ((query[0] < ':' && query[0] > '/') || query[0] == '-' ||
                (query[0] == '.' && query.length > 1 && (query[1] < ':' && query[1] > '/')))
        {
            Object val;
            try
            {
                String s = new String(query, 0, query.length);
                if (s.indexOf('.') >= 0)
                {
                    val = new Double(s);
                }
                else
                {
                    val = new Integer(s);
                }
                QuerySegment qs = new QuerySegment(QuerySegment.NUMBER);
                qs.addValue(val);
                add(qs);
                return;
            }
            catch (NumberFormatException nfe)
            {
            }
        }
        // Strings

        if (query[0] == SQUOTE)
        {

            if (query[query.length - 1] != SQUOTE)

            {
                throwIllegalArgumentException(query, "missing matching end quote");
            }


            // create the new string constant

            String id = new String(query, 1, query.length - 2);

            add(new QuerySegment(id, QuerySegment.STRING));


            return;

        }

        // Always leave the query index on a token
        int queryIdx = -1;

        // Attributes
        if (query[0] == ATTR)
        {
            // search for the beginning of the next name element and create the attribute name
            for (queryIdx = 1;
                    queryIdx < query.length && query[queryIdx] != SLASH && query[queryIdx] != LBRACKET;
                    queryIdx++)
            {
            }

            String id = new String(query, 1, queryIdx - 1);
            //Category.getInstance(this.getClass()).debug("getting attribute name: " + attrName);
            add(new QuerySegment(id, QuerySegment.ATTRIBUTE));

            // if we are done at this point, don't fall through the rest of the method
            if (queryIdx == query.length)
            {
                return;
            }

            // if the index is sitting on a left bracket, back it off by one.  the next phase
            // of the parser expects slashes to be skipped, but not other elements.
            if (query[queryIdx] == LBRACKET)
            {
                queryIdx--;
            }

        }

        int lastExprIdx;
        int begParenIdx;
        int begBracketIdx;
        int endBracketIdx;
        int begBraceIdx;
        int endBraceIdx;
        int paramStart;
        int paramEnd;
        int[] paramIdxs = new int[10];
        int commaNbr;
        int parenDepth;
        int braceDepth;
        int bracketDepth;
        char c;

        do  // this loop is to search through the query elements
        {
            lastExprIdx = queryIdx;
            begParenIdx = -1;
            begBracketIdx = -1;
            endBracketIdx = -1;
            begBraceIdx = -1;
            endBraceIdx = -1;
            paramIdxs[0] = -1;
            paramIdxs[1] = -1;
            paramIdxs[2] = -1;
            paramIdxs[3] = -1;
            paramIdxs[4] = -1;
            paramIdxs[5] = -1;
            paramIdxs[6] = -1;
            paramIdxs[7] = -1;
            paramIdxs[8] = -1;
            paramIdxs[9] = -1;
            commaNbr = 0;

            parenDepth = 0;
            braceDepth = 0;
            bracketDepth = 0;

            while (true)
            {
                queryIdx++;

                if (queryIdx == query.length)
                {
                    break;
                }

                c = query[queryIdx];

                //Category.getInstance(this.getClass()).debug("queryIdx: " + queryIdx + " query position: " + c );

                if (c == SLASH)
                {

                    // check to make sure that this isn't an embedded slash
                    if ((parenDepth > 0) || (braceDepth > 0) || (bracketDepth > 0))
                    {

                        //Category.getInstance(this.getClass()).debug("skipping embedded slash at pos: " + queryIdx );
                        continue;
                    }
                    else if (queryIdx == 0)
                    {
                        //Category.getInstance(this.getClass()).debug("root query segment found");
                        add(new QuerySegment(QuerySegment.ROOT));
                        lastExprIdx = 0;
                        continue;
                    }
                    else
                    {
                        // If it finds a slash then it breaks out of the inner loop

                        //Category.getInstance(this.getClass()).debug("breaking element parse loop at pos: " + queryIdx );
                        break;
                    }

                }
                // parens
                else if (c == LPAREN)
                {

                    parenDepth++;
                    if (begParenIdx < 0 && braceDepth == 0 && bracketDepth == 0)

                    {
                        begParenIdx = queryIdx;
                    }
                }
                else if (c == RPAREN)
                {

                    parenDepth--;
                    paramIdxs[commaNbr] = queryIdx;
                }
                // brackets
                else if (c == LBRACKET)
                {

                    bracketDepth++;
                    if (begBracketIdx < 0 && parenDepth == 0 && braceDepth == 0)

                    {
                        begBracketIdx = queryIdx;
                    }
                }
                else if (c == RBRACKET)
                {

                    bracketDepth--;
                    endBracketIdx = queryIdx;
                }
                // braces
                else if (c == LBRACE)
                {

                    braceDepth++;
                    if (begBraceIdx < 0 && parenDepth == 0 && bracketDepth == 0)

                    {
                        begBraceIdx = queryIdx;
                    }
                }
                else if (c == RBRACE)
                {

                    braceDepth--;
                    endBraceIdx = queryIdx;
                }
                // single quotes
                else if (c == SQUOTE)
                {

                    // Set the queryIdx to the next matching quote

                    queryIdx = findMatchingQuote(query, queryIdx + 1);

                }
                else if (c == COMMA)
                {

                    // only look for commas that are in the current method and
                    // not embedded in a quoted string
                    if (parenDepth == 1)
                    {

                        paramIdxs[commaNbr] = queryIdx;
                        commaNbr++;
                    }
                }
            }

            //-- Expression Expansion
            if (begBraceIdx > -1)
            {
                if (endBraceIdx < 0)
                {
                    throwIllegalArgumentException(query, "missing matching brace");
                }

                String id = new String(query, begBraceIdx + 1, endBraceIdx - (begBraceIdx + 1));
                Query expandQuery = getQuery(id);
                add(new QuerySegment(id, expandQuery, QuerySegment.EXPAND));
            }
            //-- Method
            else if (begParenIdx > -1)
            {
                if (paramIdxs[0] < 0)
                {
                    throwIllegalArgumentException(query, "missing matching parenthesis");
                }

                // parse out the name of the method
                String id = new String(query, lastExprIdx + 1, begParenIdx - (lastExprIdx + 1));
                QuerySegment qs = new QuerySegment(id, QuerySegment.METHOD);

                // parse out the method parameters
                paramStart = begParenIdx + 1;
                commaNbr = -1;

                // build the parameters for the method
                while (paramIdxs[++commaNbr] > -1)
                {
                    // set the end to be one less than the token position
                    paramEnd = paramIdxs[commaNbr] - 1;

                    // trim it out any white space
                    while ((paramStart < paramEnd) && (query[paramStart] <= ' '))
                    {
                        paramStart++;
                    }
                    while ((paramStart < paramEnd) && (query[paramEnd] <= ' '))
                    {
                        paramEnd--;

                    }

                    // create the parsed parameter and add it to the query segment
                    String parameter = new String(query, paramStart, (paramEnd - paramStart) + 1).trim();
                    // If it is the first and only parameter and it is empty, then don't add it

                    if (commaNbr == 0 && paramIdxs[commaNbr + 1] <= -1 && parameter.length() == 0)
                    {
                        qs.createValues();
                    }
                    else
                    {
                        Query paramQuery = getQuery(parameter);
                        qs.addValue(paramQuery);
                    }

                    // bump the start one past the end token position
                    paramStart = paramIdxs[commaNbr] + 1;
                }

                add(qs);
            }
            //-- Parent Access
            else if (lastExprIdx + 2 < query.length &&
                    query[lastExprIdx + 1] == PERIOD &&
                    query[lastExprIdx + 2] == PERIOD)
            {
                add(new QuerySegment(QuerySegment.PARENT));
            }
            //-- Current Access
            else if (lastExprIdx + 1 < query.length &&
                    query[lastExprIdx + 1] == PERIOD)
            {
                add(new QuerySegment(QuerySegment.CURRENT));
            }
            // check to see if this will be a collection only access
            else if (lastExprIdx + 1 != begBracketIdx)
            {
                int propertyEnd;
                if (begBracketIdx > -1)
                {
                    propertyEnd = begBracketIdx;
                }
                else
                {
                    propertyEnd = queryIdx;
                }

                // Parse out the property name
                String id = new String(query, lastExprIdx + 1, propertyEnd - (lastExprIdx + 1));

                add(new QuerySegment(id, QuerySegment.PROPERTY));
            }

            //-- Collection access
            if (begBracketIdx > -1)
            {
                if (endBracketIdx < 0)
                {
                    throwIllegalArgumentException(query, "missing matching bracket");
                }

                // parse out the name of the key
                String key = new String(query, begBracketIdx + 1, endBracketIdx - (begBracketIdx + 1));
                Query queryKey = getQuery(key);
                add(new QuerySegment(key, queryKey, QuerySegment.COLLECTION));
            }
        }
        while (queryIdx < query.length);

    }

    /**
     * Return the first non blank character or ' ' if only blank characters remain in the array
     */
    private char getNextChar(char[] query, int queryIdx)
    {
        char nextChar;
        while (++queryIdx < query.length)
        {
            nextChar = query[queryIdx];
            if (nextChar != ' ')
            {
                return nextChar;
            }
        }
        return ' ';
    }

    /**
     * Split the query into several parts based on the supplied splitChar.
     *
     * @param query     The query to split
     * @param splitChar The character that signals a place to split the query
     *
     * @return null if they query was not split. Otherwise it returns a list of Strings.
     */
    private List splitQuery(char[] query, char splitChar)
    {
        ArrayList list = null;
        int pos = 0;
        int begin = 0;
        while (pos < query.length)
        {
            char c = query[pos];
            // if c is quote then skip to next quote
            // if c is one of ( [ { then skip to matching one, still handling quoting
            // if c is the splitChar then split now
            if (c == splitChar)
            {
                String leftExp = new String(query, begin, pos - begin);
                if (list == null)
                {
                    list = new ArrayList();
                }
                list.add(leftExp);
                begin = pos + 1;
            }
            else if (c == SQUOTE)
            {
                pos = findMatchingQuote(query, pos + 1);
            }
            else if (c == LBRACE)
            {
                pos = getMatchingChar(query, pos + 1, LBRACE, RBRACE);
            }
            else if (c == LBRACKET)
            {
                pos = getMatchingChar(query, pos + 1, LBRACKET, RBRACKET);
            }
            else if (c == LPAREN)
            {
                pos = getMatchingChar(query, pos + 1, LPAREN, RPAREN);
            }
            pos++;
        }
        // If begin is > 0 we had a split so now add the last part of it
        if (begin > 0)
        {
            list.add(new String(query, begin, query.length - begin));
        }
        return list;
    }

    /**
     * Return the position of the matching closeChar. This method is for example used to find the position of the
     * closing parenthesis in a method call. Quoted expressions that are found are ignored.
     */
    private int getMatchingChar(char[] query, int pos, char openChar, char closeChar)
    {
        int depth = 0;
        while (pos < query.length)
        {
            char c = query[pos];
            if (c == SQUOTE)
            {
                pos = findMatchingQuote(query, pos + 1);
            }
            else if (c == openChar)
            {
                depth++;
            }
            else if (c == closeChar)
            {
                // We found a closing character. if depth is 0 it is the matching one
                if (depth == 0)
                {
                    return pos;
                }
                else
                {
                    depth--;
                }
            }
            pos++;
        }
        throwIllegalArgumentException(query, "missing matching end character: " + closeChar);
        return -1;
    }

    /**
     * Return the position of the closing quote character. The quote handling contained bugs in version 1.4 and earlier.
     * However in some cases it would accept quotes (') inside quotes. That is not possible to handle in a bulletproof
     * way but we try our best here so as not to break backwards compatibility. The correct solution would be to have
     * some escape handling to be able to have quotes inside quotes.
     */
    private int findMatchingQuote(char query[], int pos)
    {
        while (pos < query.length)
        {
            char c = query[pos];
            if (c == SQUOTE)
            {
                // We should only close the quote if the next character is a character that
                // would be allowed after a quote. That is basically: ) ] } / , + or if there are
                // only blank characters.
                // This is not bulletproof, but better than previous versions.

                char nextChar = getNextChar(query, pos);

                // If the end of the string is reached or a , or a + then close the quote now
                if (nextChar == ' ' || nextChar == COMMA || nextChar == CONCAT)
                {
                    return pos;
                }
                // If the next character was a ) ] } then we continue looking for more of
                // those closing characters, ) ] }
                if (nextChar == RPAREN || nextChar == RBRACKET || nextChar == RBRACE)
                {
                    int closePos = pos;
                    do
                    {
                        nextChar = getNextChar(query, ++closePos);
                    }
                    while (nextChar == RPAREN || nextChar == RBRACKET || nextChar == RBRACE);
                    // We have reached the end of closing parenthesis and brackets, now check
                    // if we can close the quote
                    if (nextChar == ' ' || nextChar == COMMA || nextChar == CONCAT || nextChar == SLASH)
                    {
                        return pos;
                    }
                }
            }
            pos++;
        }
        throwIllegalArgumentException(query, "missing matching end quote");
        return -1;
    }

    // Public --------------------------------------------------------
    /**
     * Returns the parsed query segments.
     */
    public QuerySegment[] getSegments()
    {
        return segments;
    }

    /**
     * String representation of this object.
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder();
        sb.append("query=\"").append(queryString).append("\"");

        for (int i = 0; i < segments.length; i++)
        {
            QuerySegment segment = segments[i];
            if (segment != null)
            {
                sb.append(" {").append(segment).append("}");
            }
        }
        return sb.toString();
    }

    // Private -------------------------------------------------------
    /**
     * Always throws IllegalArgumentException.
     */
    private Object throwIllegalArgumentException(char[] query, String message)
    {
        throw new IllegalArgumentException(message + " for query: " + new String(query));
    }

    /**
     * Add another segment to the segments array.
     */
    private void add(QuerySegment qs)
    {
        if (segmentsIdx == segments.length - 1)
        {
            QuerySegment[] resize = new QuerySegment[segments.length + 5];
            System.arraycopy(segments, 0, resize, 0, segments.length);
            segments = resize;
        }

        segments[segmentsIdx++] = qs;
    }

}
