/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.config;

import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;

/**
 * This is a caching implementation of ConfigurationInterface. This class can be used instead of the
 * DefaultConfiguration to make your configuration lookups more efficient. At startup it iterates through all the
 * configuration settings returned by DefaultConfiguration and stores them in a Map.
 *
 * @author Dick Zetterberg (dick@transitor.se)
 * @version $Revision: 1.2 $
 */
public class CachingConfiguration extends DefaultConfiguration
{
    // Attributes ----------------------------------------------------
    protected Map configurationMap;

    // Constructors --------------------------------------------------
    public CachingConfiguration()
    {
        // Create the configuration Map with all the keys and values
        // from the DefaultConfiguration
        configurationMap = getConfigurationMap(config);
    }

    /**
     * Create and return a Map with configuration key/values
     */
    protected Map getConfigurationMap(ConfigurationInterface configObject)
    {
        // Get an iterator for the configObject
        Iterator it = configObject.listImpl();
        Map configMap = new HashMap();
        while (it.hasNext())
        {
            String key = (String) it.next();
            // Get the value for the key
            Object value = configObject.getImpl(key);
            // Put the key and value in the Map
            configMap.put(key, value);
        }
        return configMap;
    }

    /**
     * Get a named setting.
     */
    public Object getImpl(String aName)
            throws IllegalArgumentException
    {
        Object value = configurationMap.get(aName);
        if (value == null)
        {
            throw new WebworkConfigurationNotFoundException(this.getClass(), "No such setting", aName);
        }

        return value;
    }

    /**
     * Set a named setting
     */
    public void setImpl(String aName, Object aValue)
    {
        configurationMap.put(aName, aValue);
    }

    /**
     * List setting names
     */
    public Iterator listImpl()
    {
        return configurationMap.keySet().iterator();
    }
}
