/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.action;

import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import webwork.action.factory.ApplicationMap;
import webwork.action.factory.ParameterMap;
import webwork.action.factory.SessionMap;
import webwork.multipart.MultiPartRequestWrapper;
import webwork.util.ServletValueStack;

/**
 * A context used to store servlet specific information.
 *
 * @author Maurice C. Parker (maurice@vineyardenterprise.com)
 * @version $Revision: 2 $
 * @see webwork.action.ActionContext
 */
public class ServletActionContext
{
    public static final String REQUEST = "webwork.action.ServletContext.request";
    public static final String RESPONSE = "webwork.action.ServletContext.response";
    public static final String SERVLET_CONTEXT = "webwork.action.ServletContext.servletContext";

    /**
     * Returns the HttpServletRequest object.
     *
     * @return HttpServletRequest
     */
    public static HttpServletRequest getRequest()
    {
        return ActionContext.getRequest();
    }

    /**
     * Return multipart request for HttpServletRequest.
     *
     * @return MultiPartRequestWrapper
     */
    // add wrapRequest code here and throw IOException!!!
    public static MultiPartRequestWrapper getMultiPartRequest()
    {
        return ActionContext.getMultiPartRequest();
    }

    /**
     * Set the HttpServletRequest.
     */
    public static void setRequest(HttpServletRequest request)
    {
        ActionContext.setRequest(request);
    }

    /**
     * Returns the HttpServletResponse.
     *
     * @return HttpServletResponse
     */
    public static HttpServletResponse getResponse()
    {
        return ActionContext.getResponse();
    }

    /**
     * Set the HttpServletResponse.
     */
    public static void setResponse(HttpServletResponse response)
    {
        ActionContext.setResponse(response);
    }

    /**
     * Returns the ServletContext when in a servlet environment.
     *
     * @return ServletContext or null.
     */
    public static ServletContext getServletContext()
    {
        return ActionContext.getServletContext();
    }

    /**
     * Set the ServletContext.
     */
    public static void setServletContext(ServletContext context)
    {
        ActionContext.setServletContext(context);
    }

    /**
     * Set the ServletContext.
     */
    public static void setContext(HttpServletRequest request,
            HttpServletResponse response,
            ServletContext servletContext,
            String actionName)
    {
        ActionContext context = ActionContext.getContext();
        context.setNameImpl(actionName);
        context.setLocaleImpl(request.getLocale());
        context.setSessionImpl(new SessionMap(request.getSession()));
        context.setApplicationImpl(new ApplicationMap(servletContext));
        Map parameters = new ParameterMap(request);
        context.setParametersImpl(parameters);
        context.setSingleValueParametersImpl(new SingleValueMap(parameters));
        context.setPrincipalImpl(request.getUserPrincipal());

        context.setRequestImpl(request);
        context.setResponseImpl(response);
        context.setServletContextImpl(servletContext);

        // Servlets need a ServletValueStack that has the request and page context set
        // This is created here
        ServletValueStack vs = ServletValueStack.getStack(request);
        context.setValueStackImpl(vs);
    }

}
