/*
 * Decompiled with CFR 0.152.
 */
package devliving.online.securedpreferencestore;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import devliving.online.securedpreferencestore.EncryptionManager;
import devliving.online.securedpreferencestore.Logger;
import devliving.online.securedpreferencestore.RecoveryHandler;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

public class SecuredPreferenceStore
implements SharedPreferences {
    private static final int[] VERSIONS_WITH_BREAKING_CHANGES = new int[]{10};
    static final String VERSION_KEY = "VERSION";
    private static final String DEFAULT_PREF_FILE_NAME = "SPS_file";
    private final String[] RESERVED_KEYS;
    private SharedPreferences mPrefs;
    private EncryptionManager mEncryptionManager;
    private static RecoveryHandler mRecoveryHandler;
    private static SecuredPreferenceStore mInstance;

    private SecuredPreferenceStore(@NonNull Context appContext, @Nullable String storeName, @Nullable String keyPrefix, @Nullable byte[] bitShiftingKey) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException, UnrecoverableEntryException, InvalidAlgorithmParameterException, NoSuchPaddingException, InvalidKeyException, NoSuchProviderException, MigrationFailedException {
        Logger.d("Creating store instance");
        String fileName = storeName != null ? storeName : DEFAULT_PREF_FILE_NAME;
        this.mPrefs = appContext.getSharedPreferences(fileName, 0);
        int mRunningVersion = this.mPrefs.getInt(VERSION_KEY, 9);
        if (mRunningVersion < 13) {
            new MigrationHelper(appContext, storeName, keyPrefix, bitShiftingKey).migrate(mRunningVersion, 13);
        }
        this.mEncryptionManager = new EncryptionManager(appContext, this.mPrefs, keyPrefix, bitShiftingKey, new KeyStoreRecoveryNotifier(){

            @Override
            public boolean onRecoveryRequired(Exception e, KeyStore keyStore, List<String> keyAliases) {
                if (mRecoveryHandler != null) {
                    return mRecoveryHandler.recover(e, keyStore, keyAliases, SecuredPreferenceStore.this.mPrefs);
                }
                throw new RuntimeException(e);
            }
        });
        this.RESERVED_KEYS = new String[]{VERSION_KEY, "OverridingAlias", this.mEncryptionManager.IS_COMPAT_MODE_KEY_ALIAS, this.mEncryptionManager.MAC_KEY_ALIAS, this.mEncryptionManager.AES_KEY_ALIAS};
    }

    public static void setRecoveryHandler(RecoveryHandler recoveryHandler) {
        mRecoveryHandler = recoveryHandler;
    }

    public static synchronized SecuredPreferenceStore getSharedInstance() {
        if (mInstance == null) {
            throw new IllegalStateException("Must call init() before using the store");
        }
        return mInstance;
    }

    public static void init(Context appContext, @Nullable String storeName, @Nullable String keyPrefix, @Nullable byte[] bitShiftingKey, RecoveryHandler recoveryHandler) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException, UnrecoverableEntryException, InvalidAlgorithmParameterException, NoSuchPaddingException, InvalidKeyException, NoSuchProviderException, MigrationFailedException {
        if (mInstance != null) {
            Logger.w("init called when there already is a non-null instance of the class");
            return;
        }
        SecuredPreferenceStore.setRecoveryHandler(recoveryHandler);
        mInstance = new SecuredPreferenceStore(appContext, storeName, keyPrefix, bitShiftingKey);
    }

    public static void init(Context appContext, RecoveryHandler recoveryHandler) throws IOException, CertificateException, NoSuchAlgorithmException, InvalidKeyException, UnrecoverableEntryException, InvalidAlgorithmParameterException, NoSuchPaddingException, NoSuchProviderException, KeyStoreException, MigrationFailedException {
        SecuredPreferenceStore.init(appContext, DEFAULT_PREF_FILE_NAME, null, null, recoveryHandler);
    }

    public EncryptionManager getEncryptionManager() {
        return this.mEncryptionManager;
    }

    private boolean isReservedKey(String key) {
        return Arrays.asList(this.RESERVED_KEYS).contains(key);
    }

    private boolean isReservedHashedKey(String hashedKey) {
        for (String key : this.RESERVED_KEYS) {
            try {
                if (!hashedKey.equals(EncryptionManager.getHashed(key))) continue;
                return true;
            }
            catch (NoSuchAlgorithmException e) {
                Logger.e(e);
            }
            catch (UnsupportedEncodingException e) {
                Logger.e(e);
            }
        }
        return false;
    }

    public Map<String, Object> getAll() {
        Map all = this.mPrefs.getAll();
        HashMap<String, Object> dAll = new HashMap<String, Object>(all.size());
        if (all.size() > 0) {
            for (String key : all.keySet()) {
                if (key.equals(VERSION_KEY) || this.isReservedHashedKey(key)) continue;
                try {
                    Object value = all.get(key);
                    dAll.put(key, this.mEncryptionManager.decrypt((String)value));
                }
                catch (Exception e) {
                    Logger.e(e);
                }
            }
        }
        return dAll;
    }

    public String getString(String key, String defValue) {
        if (!this.isReservedKey(key)) {
            try {
                String hashedKey = EncryptionManager.getHashed(key);
                String value = this.mPrefs.getString(hashedKey, null);
                if (value != null) {
                    return this.mEncryptionManager.decrypt(value);
                }
            }
            catch (Exception e) {
                Logger.e(e);
            }
        }
        return defValue;
    }

    public Set<String> getStringSet(String key, Set<String> defValues) {
        if (!this.isReservedKey(key)) {
            try {
                String hashedKey = EncryptionManager.getHashed(key);
                Set eSet = this.mPrefs.getStringSet(hashedKey, null);
                if (eSet != null) {
                    HashSet<String> dSet = new HashSet<String>(eSet.size());
                    for (String val : eSet) {
                        dSet.add(this.mEncryptionManager.decrypt(val));
                    }
                    return dSet;
                }
            }
            catch (Exception e) {
                Logger.e(e);
            }
        }
        return defValues;
    }

    public int getInt(String key, int defValue) {
        String value = this.getString(key, null);
        if (value != null) {
            return Integer.parseInt(value);
        }
        return defValue;
    }

    public long getLong(String key, long defValue) {
        String value = this.getString(key, null);
        if (value != null) {
            return Long.parseLong(value);
        }
        return defValue;
    }

    public float getFloat(String key, float defValue) {
        String value = this.getString(key, null);
        if (value != null) {
            return Float.parseFloat(value);
        }
        return defValue;
    }

    public boolean getBoolean(String key, boolean defValue) {
        String value = this.getString(key, null);
        if (value != null) {
            return Boolean.parseBoolean(value);
        }
        return defValue;
    }

    public byte[] getBytes(String key) {
        String val = this.getString(key, null);
        if (val != null) {
            return EncryptionManager.base64Decode(val);
        }
        return null;
    }

    public boolean contains(String key) {
        try {
            String hashedKey = EncryptionManager.getHashed(key);
            return this.mPrefs.contains(hashedKey);
        }
        catch (Exception e) {
            Logger.e(e);
            return false;
        }
    }

    public Editor edit() {
        return new Editor();
    }

    public void registerOnSharedPreferenceChangeListener(SharedPreferences.OnSharedPreferenceChangeListener onSharedPreferenceChangeListener) {
        if (this.mPrefs != null) {
            this.mPrefs.registerOnSharedPreferenceChangeListener(onSharedPreferenceChangeListener);
        }
    }

    public void unregisterOnSharedPreferenceChangeListener(SharedPreferences.OnSharedPreferenceChangeListener onSharedPreferenceChangeListener) {
        if (this.mPrefs != null) {
            this.mPrefs.unregisterOnSharedPreferenceChangeListener(onSharedPreferenceChangeListener);
        }
    }

    public class MigrationFailedException
    extends Exception {
        public MigrationFailedException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    private class MigrationHelper {
        String storeName;
        String keyPrefix;
        byte[] bitShiftKey;
        Context mContext;

        public MigrationHelper(Context context, String storeName, String keyPrefix, byte[] bitShiftKey) {
            this.storeName = storeName;
            this.keyPrefix = keyPrefix;
            this.bitShiftKey = bitShiftKey;
            this.mContext = context;
        }

        void migrateToV10() throws MigrationFailedException {
            SharedPreferences prefToWrite;
            if (this.storeName == null && this.keyPrefix == null && this.bitShiftKey == null) {
                return;
            }
            SharedPreferences prefToRead = prefToWrite = this.mContext.getSharedPreferences(SecuredPreferenceStore.DEFAULT_PREF_FILE_NAME, 0);
            boolean filenameChanged = false;
            boolean prefixChanged = false;
            if (this.storeName != null && !this.storeName.equals(SecuredPreferenceStore.DEFAULT_PREF_FILE_NAME)) {
                prefToWrite = this.mContext.getSharedPreferences(this.storeName, 0);
                filenameChanged = true;
            }
            String storedPrefix = null;
            try {
                storedPrefix = prefToWrite.getString(EncryptionManager.getHashed("OverridingAlias"), null);
            }
            catch (NoSuchAlgorithmException e) {
                throw new MigrationFailedException("Migration to Version: 0.7.0: Failed to hash a key", e);
            }
            catch (UnsupportedEncodingException e) {
                throw new MigrationFailedException("Migration to Version: 0.7.0: Failed to hash a key", e);
            }
            boolean bl = prefixChanged = storedPrefix == null && this.keyPrefix != null && !this.keyPrefix.equals("sps");
            if ((filenameChanged || prefixChanged) && prefToRead.getAll().size() > 0) {
                try {
                    EncryptionManager readCrypto = new EncryptionManager(this.mContext, prefToRead, null);
                    EncryptionManager writeCrypto = new EncryptionManager(this.mContext, prefToWrite, this.keyPrefix, this.bitShiftKey, null);
                    Map allData = prefToRead.getAll();
                    SharedPreferences.Editor editor = prefToWrite.edit();
                    for (Map.Entry entry : allData.entrySet()) {
                        String hashedKey = (String)entry.getKey();
                        if (hashedKey.equals(EncryptionManager.getHashed(readCrypto.AES_KEY_ALIAS)) || hashedKey.equals(EncryptionManager.getHashed(readCrypto.IS_COMPAT_MODE_KEY_ALIAS)) || hashedKey.equals(EncryptionManager.getHashed(readCrypto.MAC_KEY_ALIAS)) || entry.getValue() == null) continue;
                        if (entry.getValue() instanceof Set) {
                            Set values = (Set)entry.getValue();
                            HashSet<String> eValues = new HashSet<String>();
                            for (String value : values) {
                                String dValue = readCrypto.decrypt(value);
                                eValues.add(writeCrypto.encrypt(dValue));
                            }
                            editor.putStringSet(hashedKey, eValues);
                            continue;
                        }
                        if (entry.getValue() instanceof String) {
                            String dValue = readCrypto.decrypt((String)entry.getValue());
                            editor.putString(hashedKey, writeCrypto.encrypt(dValue));
                            continue;
                        }
                        Logger.e("Found a value that is not String or Set, key: " + hashedKey + ", value: " + entry.getValue());
                    }
                    if (editor.commit()) {
                        editor.putInt(SecuredPreferenceStore.VERSION_KEY, 10).apply();
                        this.cleanupPref(SecuredPreferenceStore.DEFAULT_PREF_FILE_NAME);
                    }
                }
                catch (InvalidKeyException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (UnrecoverableEntryException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (KeyStoreException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (NoSuchAlgorithmException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (NoSuchProviderException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (CertificateException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (UnsupportedEncodingException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (InvalidAlgorithmParameterException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (EncryptionManager.InvalidMacException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (IllegalBlockSizeException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (BadPaddingException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (NoSuchPaddingException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
                catch (IOException e) {
                    throw new MigrationFailedException("Migration to Version: 0.7.0: Encryption/Hashing Error", e);
                }
            }
        }

        void migrate(int fromVersion, int toVersion) throws MigrationFailedException {
            if (fromVersion >= toVersion) {
                return;
            }
            for (int version : VERSIONS_WITH_BREAKING_CHANGES) {
                if (fromVersion >= version) continue;
                this.migrate(version);
                fromVersion = version;
            }
            SecuredPreferenceStore.this.mPrefs.edit().putInt(SecuredPreferenceStore.VERSION_KEY, toVersion).apply();
        }

        void migrate(int toVersion) throws MigrationFailedException {
            if (toVersion == 10) {
                Logger.d("Migrating to: " + toVersion);
                this.migrateToV10();
            }
        }

        void cleanupPref(String storeName) {
            SharedPreferences prefs = this.mContext.getSharedPreferences(storeName, 0);
            if (prefs.getAll().size() > 0) {
                prefs.edit().clear().commit();
            }
            if (Build.VERSION.SDK_INT >= 24) {
                this.mContext.deleteSharedPreferences(storeName);
            } else {
                try {
                    new File(this.mContext.getCacheDir().getParent() + "/shared_prefs/" + storeName + ".xml").delete();
                }
                catch (Exception e) {
                    Logger.w("Unable to remove store file completely");
                }
            }
        }
    }

    public static interface KeyStoreRecoveryNotifier {
        public boolean onRecoveryRequired(Exception var1, KeyStore var2, List<String> var3);
    }

    public class Editor
    implements SharedPreferences.Editor {
        SharedPreferences.Editor mEditor;

        public Editor() {
            this.mEditor = SecuredPreferenceStore.this.mPrefs.edit();
        }

        public SharedPreferences.Editor putString(String key, String value) {
            if (SecuredPreferenceStore.this.isReservedKey(key)) {
                Logger.e("Trying to store value for a reserved key, value: " + value);
                return this;
            }
            try {
                String hashedKey = EncryptionManager.getHashed(key);
                String evalue = SecuredPreferenceStore.this.mEncryptionManager.encrypt(value);
                this.mEditor.putString(hashedKey, evalue);
            }
            catch (Exception e) {
                Logger.e(e);
            }
            return this;
        }

        public SharedPreferences.Editor putStringSet(String key, Set<String> values) {
            if (SecuredPreferenceStore.this.isReservedKey(key)) {
                Logger.e("Trying to store value for a reserved key, value: " + values);
                return this;
            }
            try {
                String hashedKey = EncryptionManager.getHashed(key);
                HashSet<String> eSet = new HashSet<String>(values.size());
                for (String val : values) {
                    eSet.add(SecuredPreferenceStore.this.mEncryptionManager.encrypt(val));
                }
                this.mEditor.putStringSet(hashedKey, eSet);
            }
            catch (Exception e) {
                Logger.e(e);
            }
            return this;
        }

        public SharedPreferences.Editor putInt(String key, int value) {
            String val = Integer.toString(value);
            return this.putString(key, val);
        }

        public SharedPreferences.Editor putLong(String key, long value) {
            String val = Long.toString(value);
            return this.putString(key, val);
        }

        public SharedPreferences.Editor putFloat(String key, float value) {
            String val = Float.toString(value);
            return this.putString(key, val);
        }

        public SharedPreferences.Editor putBoolean(String key, boolean value) {
            String val = Boolean.toString(value);
            return this.putString(key, val);
        }

        public SharedPreferences.Editor putBytes(String key, byte[] bytes) {
            if (bytes != null) {
                String val = EncryptionManager.base64Encode(bytes);
                return this.putString(key, val);
            }
            return this.remove(key);
        }

        public SharedPreferences.Editor remove(String key) {
            if (SecuredPreferenceStore.this.isReservedKey(key)) {
                Logger.e("Trying to remove value for a reserved key");
                return this;
            }
            try {
                String hashedKey = EncryptionManager.getHashed(key);
                this.mEditor.remove(hashedKey);
            }
            catch (Exception e) {
                Logger.e(e);
            }
            return this;
        }

        public SharedPreferences.Editor clear() {
            for (String key : SecuredPreferenceStore.this.mPrefs.getAll().keySet()) {
                if (key.equals(SecuredPreferenceStore.VERSION_KEY) || SecuredPreferenceStore.this.isReservedHashedKey(key)) continue;
                this.mEditor.remove(key);
            }
            return this;
        }

        public boolean commit() {
            return this.mEditor.commit();
        }

        public void apply() {
            this.mEditor.apply();
        }
    }
}

