package nu.annat.beholder

import android.util.SparseArray
import android.view.ViewGroup
import androidx.lifecycle.LifecycleOwner
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView
import nu.annat.beholder.BeholderListAdapter
import nu.annat.beholder.action.ActionHandler
import nu.annat.beholder.presenter.ComponentInfo

/**
 * Helper to use components directly as RecycleView item objects.
 */
open class BeholderListAdapter<T : ComponentInfo?>(
        protected var factory: ComponentFactory,
        data: MutableList<T>,
        diffCallback: DiffUtil.ItemCallback<T>,
        protected val actionHandler: ActionHandler<*>?,
        protected val lifecycleOwner: LifecycleOwner?,
        protected val baseDepth: Int = 0
) : ListAdapter<T, ComponentViewHolder<*, *, *>>(diffCallback) {
    protected var cachedPresenters = SparseArray<ComponentInfo>()

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ComponentViewHolder<*, *, *> {
        return factory.createReusable(cachedPresenters[viewType], baseDepth, parent, lifecycleOwner, actionHandler)
    }

    override fun getItemViewType(position: Int): Int {
        val componentInfo = getItem(position)!!
        val deepLayoutHash = componentInfo.deepLayoutHash()
        cachedPresenters.put(deepLayoutHash, componentInfo)
        return deepLayoutHash
    }

    override fun getItemId(position: Int): Long {
        return getItem(position)?.itemId() ?: RecyclerView.NO_ID
    }

    override fun onBindViewHolder(holder: ComponentViewHolder<*, *, *>, position: Int) {
        factory.bindDeep(holder, position, getItem(position), true)
    }

    override fun onViewAttachedToWindow(holder: ComponentViewHolder<*, *, *>) {
        super.onViewAttachedToWindow(holder)
        holder.attached()
    }

    override fun onViewDetachedFromWindow(holder: ComponentViewHolder<*, *, *>) {
        super.onViewDetachedFromWindow(holder)
        holder.detached()
    }


    companion object {
        private val TAG = BeholderListAdapter::class.java.simpleName
    }

    init {
        submitList(data)
    }
}
