package nl.lockhead.lpf.plugins.plugin;

import nl.lockhead.lpf.exceptions.InvalidPluginException;
import nl.lockhead.lpf.logging.LPFLogger;
import nl.lockhead.lpf.plugins.PluginManager;
import org.jetbrains.annotations.NotNull;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

public class PluginContainer {

    private final List<Method> events = new ArrayList<>();
    private Plugin plugin;

    public PluginContainer(@NotNull Plugin plugin) throws InvalidPluginException {
        loadPlugin(plugin);
    }

    /**
     * Attempt to load this plugin.
     *
     * @param plugin the plugin
     * @throws InvalidPluginException thrown by {@link Plugin#load}
     */
    public void loadPlugin(@NotNull Plugin plugin) throws InvalidPluginException {
        if (plugin.isLoaded()) {
            System.err.println("The plugin is already loaded!");
            return;
        }
        plugin.load();
        plugin.setContainer(this);
        this.plugin = plugin;
    }

    /**
     * Attempt to unload this plugin.
     */
    public void unloadPlugin() {
        if (!plugin.isLoaded()) {
            System.err.println("The plugin is already unloaded!");
            return;
        }
        plugin.unload();
        PluginManager.get().getPlugins().remove(this);
        LPFLogger.getLogger(PluginManager.get()).info("Unloaded " + plugin.getConfig().getName() + " v" + plugin.getConfig().getVersion());
    }

    /**
     * Attempt to enable this plugin.
     */
    public void enablePlugin() {
        if (!plugin.isLoaded()) {
            System.err.println("The plugin is not loaded yet!");
            return;
        }
        if (plugin.isEnabled()) {
            System.err.println("The plugin is already enabled!");
            return;
        }
        plugin.enable();
        LPFLogger.getLogger(PluginManager.get()).info("Enabled " + plugin.getConfig().getName() + " v" + plugin.getConfig().getVersion());
    }

    /**
     * Attempt to disable this plugin.
     */
    public void disablePlugin() {
        if (!plugin.isLoaded()) {
            System.err.println("The plugin is not loaded yet!");
            return;
        }
        if (!plugin.isEnabled()) {
            System.err.println("The plugin is already disabled!");
            return;
        }
        plugin.disable();
        LPFLogger.getLogger(PluginManager.get()).info("Disabled " + plugin.getConfig().getName() + " v" + plugin.getConfig().getVersion());
    }

    public List<Method> getEventMethods() {
        return new ArrayList<>(events);
    }

    public List<Method> addEventMethod(Method eventMethod) {
        events.add(eventMethod);
        return getEventMethods();
    }

    public final Plugin getPlugin() {
        return plugin;
    }
}
