package nl.lockhead.lpf.plugins.plugin;

import nl.lockhead.lpf.events.LPFEventHandler;
import nl.lockhead.lpf.events.builtin.PluginDisableEvent;
import nl.lockhead.lpf.events.builtin.PluginEnableEvent;
import nl.lockhead.lpf.exceptions.InvalidPluginException;
import nl.lockhead.lpf.logging.LPFLogger;
import nl.lockhead.lpf.plugins.PluginManager;
import nl.lockhead.lpf.plugins.annotations.PluginInfo;
import nl.lockhead.lpf.plugins.threads.LPFThread;

public abstract class Plugin {

    private static Integer idCounter = 0;
    private final Integer id;
    private PluginConfig config;
    private transient PluginContainer container;
    private boolean enabled, loaded, eventsRegistered;

    protected Plugin() {
        id = idCounter++;
    }

    /**
     * Invoked when loading the plugin
     */
    public void onLoad() {
    }

    /**
     * Invoked when unloading the plugin
     */
    public void onUnload() {
    }

    /**
     * Invoked when enabling the plugin
     */
    public abstract void onEnable();

    /**
     * Invoked when disabling the plugin
     */
    public abstract void onDisable();

    /**
     * Handle further loading of the plugin, after its classes have been loaded.
     *
     * @throws InvalidPluginException thrown by {@link #loadConfig}
     */
    protected final void load() throws InvalidPluginException {
        if (loaded) {
            LPFLogger.getLogger(this).warning("Plugin " + config.getName() + " failed to load: Plugin was already loaded.");
            return;
        }
        config = loadConfig();
        LPFThread thread = new LPFThread(this::onLoad, LPFThread.ThreadType.LOAD);
        thread.start();
        loaded = true;
        try {
            if (config.shouldRegisterEvents())
                registerEvents();
        } catch (IllegalStateException e) {
            e.printStackTrace();
            loaded = false;
        }
    }

    /**
     * Load the plugin's config. Uses the {@link PluginInfo} annotation.
     * @return the generated {@link PluginConfig} object
     * @throws InvalidPluginException when no {@link PluginInfo} annotation is found.
     */
    @SuppressWarnings("unchecked")
    private PluginConfig loadConfig() throws InvalidPluginException {
        Class<Plugin> obj = (Class<Plugin>) this.getClass();
        if (obj.isAnnotationPresent(PluginInfo.class)) {
            PluginInfo a = obj.getAnnotation(PluginInfo.class);
            return PluginConfig.createConfig(a.name(), a.author(), Version.getVersion(a.version()));
        }
        throw new InvalidPluginException("Plugin (instanceId=" + id + ") failed to load: @PluginInfo annotation not present.");
    }

    /**
     * Search for, and register, events in this plugin.
     *
     */
    protected final void registerEvents() {
        if (eventsRegistered) {
            getLogger().warning("Plugin (instanceId=" + id + ") failed to register event listeners: Events already registered.");
            return;
        }
        if (!loaded) {
            throw new IllegalStateException("Plugin (instanceId=" + id + ") failed to register event listeners: Plugin not loaded yet.");
        }
        PluginManager.get().registerEvents(this);
        eventsRegistered = true;
    }

    /**
     * Unload this plugin, and unregister its events.
     *
     */
    public final void unload() {
        if (enabled) {
            LPFLogger.getLogger(this).warning("Plugin (name=" + config.getName() + ") failed to unload: Plugin needs to be disabled first.");
            return;
        }
        if (!loaded) {
            LPFLogger.getLogger(this).warning("Plugin (instanceId=" + id + ") failed to unload: Plugin was already unloaded.");
            return;
        }
        LPFThread thread = new LPFThread(this::onUnload, LPFThread.ThreadType.UNLOAD);
        thread.start();
        if (eventsRegistered) {
            for (PluginContainer pc : PluginManager.get().getRegisteredEvents().keySet()) {
                if (pc.getPlugin().getId() == id) {
                    PluginManager.get().getRegisteredEvents().remove(pc);
                    break;
                }
            }
            eventsRegistered = false;
        }
        loaded = false;
    }

    /**
     * Enable this plugin, registering its event listeners.
     *
     */
    public final void enable() {
        if (!loaded) {
            LPFLogger.getLogger(this).warning("Plugin (instanceId=" + id + ") failed to enable: Plugin not loaded yet.");
            return;
        }
        if (enabled) {
            LPFLogger.getLogger(this).warning("Plugin " + config.getName() + " failed to enable: Plugin already enabled.");
            return;
        }
        LPFEventHandler.getLPFEventHandler().handleEvent(new PluginEnableEvent(this));
        LPFThread thread = new LPFThread(this::onEnable, LPFThread.ThreadType.ENABLE);
        thread.start();
        enabled = true;
    }

    /**
     * Disable this plugin, removing any registered event listeners.
     *
     */
    public final void disable() {
        if (!loaded) {
            LPFLogger.getLogger(this).warning("Plugin (instanceId=" + id + ") failed to disable: Plugin not loaded yet.");
            return;
        }
        if (!enabled) {
            LPFLogger.getLogger(this).warning("Plugin " + config.getName() + " failed to disable: Plugin already disabled.");
            return;
        }
        LPFEventHandler.getLPFEventHandler().handleEvent(new PluginDisableEvent(this));
        LPFThread thread = new LPFThread(this::onDisable, LPFThread.ThreadType.DISABLE);
        thread.start();
        enabled = false;
    }

    public final boolean isLoaded() {
        return loaded;
    }

    public final boolean isEnabled() {
        return enabled;
    }

    public final boolean isEventsRegistered() {
        return eventsRegistered;
    }

    public final int getId() {
        return id;
    }

    protected final LPFLogger getLogger() {
        return LPFLogger.getLogger(this);
    }

    public PluginConfig getConfig() {
        return config;
    }

    public PluginContainer getContainer() {
        return container;
    }

    protected void setContainer(PluginContainer container) {
        this.container = container;
    }
}
