package nl.lockhead.lpf.plugins.loaders.impl;

import nl.lockhead.lpf.plugins.loaders.IPluginLoader;
import nl.lockhead.lpf.plugins.plugin.Plugin;
import nl.lockhead.lpf.plugins.plugin.PluginContainer;
import nl.lockhead.lpf.tools.FileManager;

import java.io.*;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;

public class URLPluginLoader implements IPluginLoader {

    protected final URL[] urls;
    protected boolean running;
    protected List<URL> classloaderUrls;

    public URLPluginLoader(List<URL> classloaderUrls, URL... urls) {
        this.classloaderUrls = classloaderUrls;
        this.urls = urls;
    }

    @Override
    public Set<PluginContainer> loadPlugins() {
        return loadPlugins(urls);
    }

    protected Set<PluginContainer> loadPlugins(URL... urls) {
        if (!running) {
            running = true;
            Set<PluginContainer> loaded = new HashSet<>();
            for (URL url : urls) {
                try {
                    PluginContainer pc = new PluginContainer(getPlugin(url));
                    loaded.add(pc);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            return loaded;
        } else
            System.err.print("A plugin loader instance is already running!");
        return Collections.emptySet();
    }

    /**
     * Make a connection to the {@link #urls} and load the classes from the connection's input stream.
     * @param url the URL from which to download
     *
     * @return hopefully a {@link Plugin}
     * @throws IOException hence, hopefully
     */
    protected Plugin getPlugin(URL url) throws IOException {
        URLConnection conn = url.openConnection();
        conn.setConnectTimeout(3000);
        conn.addRequestProperty("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:56.0) Gecko/20100101 Firefox/56.0");
        conn.connect();
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        InputStream is = conn.getInputStream();
        int nRead;
        byte[] data = new byte[16384];

        while ((nRead = is.read(data, 0, data.length)) != -1) {
            buffer.write(data, 0, nRead);
        }
        return readBuffer(buffer.toByteArray());
    }

    /**
     * https://stackoverflow.com/questions/28964450/loading-a-jar-dynamically-from-memory
     *
     * @param buffer the byte buffer from which to load JarEntries
     * @return a Plugin if everything goes right
     * @throws IOException if not everything goes right
     */
    protected Plugin readBuffer(byte[] buffer) throws IOException {
        final Map<String, byte[]> map = new HashMap<>();
        try (JarInputStream is = new JarInputStream(new ByteArrayInputStream(buffer))) {
            for (; ; ) {
                JarEntry nextEntry = is.getNextJarEntry();
                if (nextEntry == null)
                    break;
                final int est = (int) nextEntry.getSize();
                byte[] data = new byte[est > 0 ? est : 1024];
                int real = 0;
                for (int r = is.read(data); r > 0; r = is.read(data, real, data.length - real))
                    if (data.length == (real += r))
                        data = Arrays.copyOf(data, data.length * 2);
                if (real != data.length)
                    data = Arrays.copyOf(data, real);
                map.put("/" + nextEntry.getName(), data);
            }
        }
        URL u = new URL("x-buffer", null, -1, "/", new URLStreamHandler() {
            protected URLConnection openConnection(URL u) throws IOException {
                final byte[] data = map.get(u.getFile());

                if (data == null)
                    throw new FileNotFoundException(u.getFile());

                return new URLConnection(u) {

                    public void connect() {
                    }

                    @Override
                    public InputStream getInputStream() {
                        return new ByteArrayInputStream(data);
                    }
                };
            }
        });
        Plugin plugin = null;
        URL[] urls = Arrays.copyOf(classloaderUrls.toArray(new URL[0]), classloaderUrls.size() + 2);
        urls[urls.length - 2] = u;
        urls[urls.length - 1] = new URL("jar:file:" + Objects.requireNonNull(FileManager.getThisFile()).getAbsolutePath() + "!/");
        try (URLClassLoader cl = URLClassLoader.newInstance(urls, Plugin.class.getClassLoader())) {
            for (String s : map.keySet()) {
                if (s.endsWith(".class")) {
                    String classpath = s.substring(1).replace('/', '.').replace(".class", "");
                    try {
                        Class<?> c = cl.loadClass(classpath);
                        if (c.getSuperclass() == Plugin.class)
                            plugin = (Plugin) c.getConstructor().newInstance();
                    } catch (Exception e) {
                        System.out.println("s = " + s);
                        e.printStackTrace();
                    }
                }
            }

            return plugin;
        }
    }

    public URL[] getUrls() {
        return urls;
    }

    @Override
    public List<URL> getClassloaderUrls() {
        return classloaderUrls;
    }

    @Override
    public boolean isRunning() {
        return running;
    }
}
