package nl.lockhead.lpf.plugins.loaders.impl;

import nl.lockhead.lpf.exceptions.InvalidPluginException;
import nl.lockhead.lpf.plugins.loaders.IPluginLoader;
import nl.lockhead.lpf.plugins.plugin.Plugin;
import nl.lockhead.lpf.plugins.plugin.PluginContainer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Collectors;

public class FilePluginLoader implements IPluginLoader {

    private final File directory;
    private final boolean subDirs;
    private boolean running;
    private List<URL> classloaderUrls;

    public FilePluginLoader(List<URL> classloaderUrls, File directory, boolean checkSubDirectories) throws IllegalArgumentException {
        if (!directory.isDirectory())
            throw new IllegalArgumentException("\"" + directory.getAbsolutePath() + "\" is not a directory.");
        this.directory = directory;
        subDirs = checkSubDirectories;
        this.classloaderUrls = classloaderUrls;
    }

    @Override
    public Set<PluginContainer> loadPlugins() {
        return loadPlugins(directory, subDirs);
    }

    public Set<PluginContainer> loadPlugins(File directory, boolean checkSubDirectories) {
        if (!running) {
            running = true;
            try {
                Set<PluginContainer> loaded = new HashSet<>();
                search(plugins -> {
                    for (File f : plugins) {
                        try {
                            PluginContainer pc = new PluginContainer(loadPlugin(f));
                            loaded.add(pc);
                        } catch (InvalidPluginException | IOException e) {
                            e.printStackTrace();
                        }
                    }
                    running = false;
                }, directory, checkSubDirectories);
                return loaded;
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else
            System.err.print("A plugin loader instance is already running!");
        return Collections.emptySet();
    }

    public Plugin loadPlugin(File file) throws InvalidPluginException, IOException {
        if (!file.exists())
            throw new FileNotFoundException(file.getAbsolutePath() + " doesn't exist.");
        if (!file.isFile() || !file.getName().endsWith(".jar"))
            throw new InvalidPluginException(file.getAbsolutePath() + " isn't a Jar file.");
        Plugin plugin = null;
        try (JarFile jarFile = new JarFile(file)) {
            Enumeration<JarEntry> e = jarFile.entries();
            URL[] urls = Arrays.copyOf(classloaderUrls.toArray(new URL[0]), classloaderUrls.size() + 1);
            urls[urls.length - 1] = new URL("jar:file:" + file.getAbsolutePath() + "!/");
            try (URLClassLoader cl = URLClassLoader.newInstance(urls, Plugin.class.getClassLoader())) {
                while (e.hasMoreElements()) {
                    JarEntry je = e.nextElement();
                    if (je.isDirectory() || !je.getName().endsWith(".class")) {
                        continue;
                    }
                    String className = je.getName().substring(0, je.getName().length() - 6);
                    className = className.replace('/', '.');
                    Class<?> c = cl.loadClass(className);
                    if (c.getSuperclass() == Plugin.class)
                        plugin = (Plugin) c.getConstructor().newInstance();
                }
                return plugin;
            }
        } catch (Exception e1) {
            e1.printStackTrace();
        }
        throw new InvalidPluginException("null");
    }

    public void search(Callback callback, File directory, boolean subDirs) throws IOException {
        List<File> pluginsFound = new ArrayList<>();
        List<File> f = (subDirs ? Files.walk(directory.toPath()) : Files.list(directory.toPath()))
                .filter(Files::isRegularFile)
                .map(Path::toFile)
                .collect(Collectors.toList());
        f.stream().filter(file -> file.getName().endsWith(".jar")).forEach(pluginsFound::add);
        callback.onFinish(pluginsFound);
    }

    public final File getDirectory() {
        return directory;
    }

    public final boolean isSubDirs() {
        return subDirs;
    }

    @Override
    public boolean isRunning() {
        return running;
    }

    @Override
    public List<URL> getClassloaderUrls() {
        return classloaderUrls;
    }

    public interface Callback {
        void onFinish(List<File> plugins);
    }
}
