package nl.lockhead.lpf.plugins.plugin;

import nl.lockhead.lpf.exceptions.InvalidPluginException;
import nl.lockhead.lpf.logging.LPFLogger;
import nl.lockhead.lpf.plugins.PluginManager;
import nl.lockhead.lpf.plugins.annotations.LoggerSettings;
import org.jetbrains.annotations.NotNull;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

@LoggerSettings(displayName = "Plugin Manager")
public class PluginContainer {

    private final List<Method> events = new ArrayList<>();
    private Plugin plugin;

    public PluginContainer(@NotNull Plugin plugin) throws InvalidPluginException {
        loadPlugin(plugin);
    }

    /**
     * Attempt to load this plugin.
     *
     * @param plugin the plugin
     * @throws InvalidPluginException thrown by {@link Plugin#load}
     */
    public void loadPlugin(@NotNull Plugin plugin) throws InvalidPluginException {
        if (plugin.isLoaded()) {
            System.err.println("The plugin is already loaded!");
            return;
        }
        plugin.load();
        plugin.setContainer(this);
        LPFLogger.getLogger(this).info("Loaded " + plugin.getConfig().getName() + " v" + plugin.getConfig().getVersion());
    }

    /**
     * Attempt to unload this plugin.
     */
    public void unloadPlugin() {
        if (!plugin.isLoaded()) {
            System.err.println("The plugin is already unloaded!");
            return;
        }
        plugin.unload();
        PluginManager.get().getPlugins().remove(this);
        LPFLogger.getLogger(this).info("Unloaded " + plugin.getConfig().getName() + " v" + plugin.getConfig().getVersion());
    }

    /**
     * Attempt to enable this plugin.
     */
    public void enablePlugin() {
        if (!plugin.isLoaded()) {
            System.err.println("The plugin is not loaded yet!");
            return;
        }
        if (plugin.isEnabled()) {
            System.err.println("The plugin is already enabled!");
            return;
        }
        plugin.enable();
        LPFLogger.getLogger(this).info("Enabled " + plugin.getConfig().getName() + " v" + plugin.getConfig().getVersion());
    }

    /**
     * Attempt to disable this plugin.
     */
    public void disablePlugin() {
        if (!plugin.isLoaded()) {
            System.err.println("The plugin is not loaded yet!");
            return;
        }
        if (!plugin.isEnabled()) {
            System.err.println("The plugin is already disabled!");
            return;
        }
        plugin.disable();
        LPFLogger.getLogger(this).info("Disabled " + plugin.getConfig().getName() + " v" + plugin.getConfig().getVersion());
    }

    public List<Method> getEventMethods() {
        return new ArrayList<>(events);
    }

    public List<Method> addEventMethod(Method eventMethod) {
        events.add(eventMethod);
        return getEventMethods();
    }

    public final Plugin getPlugin() {
        return plugin;
    }
}
