package nl.lockhead.lpf.plugins.plugin;

import nl.lockhead.lpf.exceptions.InvalidPluginException;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import static nl.lockhead.lpf.tools.Matching.MATCH_64;

/**
 * Contains the plugin's metadata, e.g. name, author, version
 */
@SuppressWarnings("unused")
public class PluginConfig {

    private final Map<String, Object> meta = new LinkedHashMap<>();
    private String name, author;
    private boolean registerEvents;
    private Version version;

    private PluginConfig(String name, String author, Version version) {
        this(name, author, version, true);
    }

    private PluginConfig(String name, String author, Version version, boolean registerEvents) {
        this.name = name;
        this.author = author;
        this.registerEvents = registerEvents;
        this.version = version;
    }

    /**
     * Generate a PluginConfig with the following values. <b>Assumes its events should be registered.</b>
     *
     * @param pluginName   the name of the plugin
     * @param pluginAuthor the plugin's author
     * @param version      the version of the plugin
     * @return the generated PluginConfig
     */
    public static PluginConfig createConfig(String pluginName, String pluginAuthor, Version version) {
        try {
            if (pluginName == null)
                throw new InvalidPluginException("Invalid plugin name: null");
            pluginName = pluginName.trim();
            if (pluginAuthor == null)
                throw new InvalidPluginException("Invalid plugin author name: null");
            pluginAuthor = pluginAuthor.trim();
            if (version == null)
                throw new InvalidPluginException("Invalid version: null");
            if (!MATCH_64.matcher(pluginName).matches())
                throw new InvalidPluginException("Invalid plugin name: Plugin names are to be alphanumeric and between 3 and 64 characters");
            if (!MATCH_64.matcher(pluginAuthor).matches())
                throw new InvalidPluginException("Invalid plugin name: Plugin Author names are to be alphanumeric and between 3 and 64 characters");
        } catch (InvalidPluginException e) {
            e.printStackTrace();
            return null;
        }
        return new PluginConfig(pluginName, pluginAuthor, version);
    }

    /**
     * Generate a PluginConfig with the following values.
     * @param pluginName the name of the plugin
     * @param pluginAuthor the plugin's author
     * @param version the version of the plugin
     * @param registerEvents whether event listeners should be looked for, and registered
     * @return the generated PluginConfig
     */
    public static PluginConfig createConfig(String pluginName, String pluginAuthor, Version version, boolean registerEvents) {
        try {
            if (pluginName == null)
                throw new InvalidPluginException("Invalid plugin name: null");
            pluginName = pluginName.trim();
            if (pluginAuthor == null)
                throw new InvalidPluginException("Invalid plugin author name: null");
            pluginAuthor = pluginAuthor.trim();
            if (version == null)
                throw new InvalidPluginException("Invalid version: null");
            if (!MATCH_64.matcher(pluginName).matches())
                throw new InvalidPluginException("Invalid plugin name: Plugin names are to be alphanumeric and between 3 and 64 characters");
            if (!MATCH_64.matcher(pluginAuthor).matches())
                throw new InvalidPluginException("Invalid plugin name: Plugin Author names are to be alphanumeric and between 3 and 64 characters");
        } catch (InvalidPluginException e) {
            e.printStackTrace();
            return null;
        }
        return new PluginConfig(pluginName, pluginAuthor, version, registerEvents);
    }

    /**
     *
     * @param pluginName the name
     * @throws InvalidPluginException if pluginName is null, or if it contains any non-alphanumeric characters, or if {@code length < 3 || > 64}
     */
    public final void setPluginName(String pluginName) throws InvalidPluginException {
        if (pluginName == null)
            throw new InvalidPluginException("Invalid plugin name: null");
        if (!MATCH_64.matcher(pluginName).matches())
            throw new InvalidPluginException("Invalid plugin name: Plugin names are to be alphanumeric and between 3 and 64 characters");
        pluginName = pluginName.trim();
        name = pluginName;
    }

    /**
     *
     * @param authorName the author name
     * @throws InvalidPluginException if authorName is null, or if it contains any non-alphanumeric characters, or if {@code length < 3 || > 64}
     */
    public final void setPluginAuthor(String authorName) throws InvalidPluginException {
        if (authorName == null)
            throw new InvalidPluginException("Invalid plugin author name: null");
        if (!MATCH_64.matcher(authorName).matches())
            throw new InvalidPluginException("Invalid plugin name: Plugin Author names are to be alphanumeric and between 3 and 64 characters");
        authorName = authorName.trim();
        author = authorName;
    }

    /**
     *
     * @param pluginVersion the {@link Version}
     * @throws InvalidPluginException if the given Version is null
     */
    public final void setPluginVersion(Version pluginVersion) throws InvalidPluginException {
        if (version == null)
            throw new InvalidPluginException("Invalid version: null");
        version = pluginVersion;
    }

    public final String getName() {
        return name;
    }

    public final String getAuthor() {
        return author;
    }

    public final boolean shouldRegisterEvents() {
        return registerEvents;
    }

    public Version getVersion() {
        return version;
    }

    public void addMetadata(String key, Object value) {
        meta.put(key, value);
    }

    public void removeMetadata(String key) {
        meta.remove(key);
    }

    public Map<String, Object> getMetadataMap() {
        return new HashMap<>(meta);
    }

}
