package nl.lockhead.lpf.plugins.loaders.impl;

import nl.lockhead.lpf.exceptions.InvalidPluginException;
import nl.lockhead.lpf.plugins.plugin.Plugin;
import nl.lockhead.lpf.plugins.plugin.PluginContainer;

import java.net.URL;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

public class AsyncURLPluginLoader extends URLPluginLoader {

    private ExecutorService executor;

    public AsyncURLPluginLoader(List<URL> classloaderUrls, URL... urls) {
        super(classloaderUrls, urls);
        executor = Executors.newFixedThreadPool(Math.min(16, Math.max(1, urls.length / 4)));
    }

    @Override
    public Set<PluginContainer> loadPlugins() {
        return loadPlugins(urls);
    }

    public Set<PluginContainer> loadPlugins(URL... urls) {
        if (!running) {
            running = true;
            List<Callable<PluginContainer>> callables = new ArrayList<>();

            Arrays.stream(urls).forEach(url -> callables.add(() -> {
                try {
                    return new PluginContainer(getPlugin(url));
                } catch (InvalidPluginException e) {
                    e.printStackTrace();
                }
                return null;
            }));

            try {
                Set<PluginContainer> result = executor.invokeAll(callables).stream()
                        .map(future -> {
                            try {
                                return future.get();
                            } catch (InterruptedException | ExecutionException e) {
                                e.printStackTrace();
                                return null;
                            }
                        }).filter(Objects::nonNull)
                        .collect(Collectors.toSet());
                running = false;
                return result;
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        } else
            System.err.print("A plugin loader instance is already running!");
        return Collections.emptySet();
    }

    @Override
    protected Plugin getPlugin(URL url) {
        try {
            System.out.println("url = " + url);
            return super.getPlugin(url);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    public ExecutorService getExecutor() {
        return executor;
    }
}
