package nl.lockhead.lpf.logging;

import nl.lockhead.lpf.plugins.annotations.LoggerSettings;
import nl.lockhead.lpf.plugins.plugin.Plugin;
import org.jetbrains.annotations.Nullable;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.*;

import static nl.lockhead.lpf.tools.Matching.MATCH_NEWLINE;

@SuppressWarnings("unused")
public class LPFLogger {

    private static final ConsoleHandler consoleHandler = new ConsoleHandler();
    private static Handler defaultHandler;
    private static List<LPFLogger> loggers = new ArrayList<>();
    private static LPFLogger defaultLogger;
    private static int indentation = 0;
    private final Object source;
    private Logger logger;
    private SimpleDateFormat dateFormat = new SimpleDateFormat("HH:mm:ss.SS");

    private LPFLogger(@Nullable Object source) {
        String name = "LPF";
        if (source != null) {
            if (source instanceof String) {
                name = (String) source;
            } else if (source instanceof Plugin) {
                if (((Plugin) source).getConfig() != null) {
                    name = ((Plugin) source).getConfig().getName();
                } else
                    name = "Plugin ID " + ((Plugin) source).getId();
            } else {
                Class<?> clazz = source.getClass();
                if (clazz.isAnnotationPresent(LoggerSettings.class))
                    name = clazz.getAnnotation(LoggerSettings.class).displayName();
                else
                    name = source.getClass().getSimpleName();
            }
        }
        logger = Logger.getLogger(name);
        logger.setUseParentHandlers(false);
        logger.setLevel(Level.ALL);
        if (defaultHandler != null) {
            defaultHandler.setLevel(Level.ALL);
            logger.addHandler(defaultHandler);
        }
        indentation = Math.max(indentation, name.length());
        this.source = source;
        loggers.add(this);
    }

    /**
     * Creates a new logger instance for the source, or uses an existing one.<br>
     * <b>Tip: use the annotation {@link LoggerSettings} to generate a custom prefix for the source class.</b>
     *
     * @param source the source, used for generating an identifier in logs
     * @return the generated logger
     */
    public static LPFLogger getLogger(@Nullable Object source) {
        if (source == null) {
            if (defaultLogger == null) {
                defaultLogger = new LPFLogger(null);
                defaultLogger.addHandler(consoleHandler);
            }
            return defaultLogger;
        }
        for (LPFLogger l : loggers) {
            if (l.source == null)
                continue;
            if (l.source.equals(source)) {
                return l;
            }
        }
        LPFLogger l = new LPFLogger(source);
        l.addHandler(consoleHandler);
        return l;
    }

    public static void addHandlerToAll(Handler handler) {
        if (defaultLogger == null)
            defaultLogger = new LPFLogger(null);
        defaultLogger.addHandler(handler);
        for (LPFLogger l : loggers) {
            if (l.source == null)
                continue;
            l.addHandler(handler);
        }
    }

    public static void removeLogger(LPFLogger logger) {
        loggers.remove(logger);
    }

    public static Handler getDefaultHandler() {
        return defaultHandler;
    }

    public static void setDefaultHandler(Handler defaultHandler) {
        LPFLogger.defaultHandler = defaultHandler;
    }

    public static List<LPFLogger> getLoggers() {
        return loggers;
    }

    public void removeLogger() {
        loggers.remove(this);
    }

    @SuppressWarnings("UnusedReturnValue")
    public Handler addHandler(Handler handler) {
        handler.setLevel(Level.ALL);
        handler.setFormatter(new SimpleFormatter() {
            @Override
            public String format(LogRecord record) {
                if (getSource() == null) {
                    return String.format("%s [%s:%s] %s\n",
                            dateFormat.format(new Date(record.getMillis())),
                            record.getLoggerName(), record.getSourceMethodName().toUpperCase(), record.getMessage());
                }
                if (getSource() instanceof Plugin) {
                    Plugin p = (Plugin) getSource();
                    String name;
                    if (p.getConfig() == null)
                        name = "Plugin ID " + p.getId();
                    else
                        name = p.getConfig().getName();
                    return String.format("%s [%s:%s] %s\n",
                            dateFormat.format(new Date(record.getMillis())),
                            name, record.getSourceMethodName().toUpperCase(), record.getMessage());
                }
                return String.format("%s [%s:%s] %s\n",
                        dateFormat.format(new Date(record.getMillis())),
                        record.getLoggerName(), record.getSourceMethodName().toUpperCase(), record.getMessage());
            }
        });
        logger.addHandler(handler);
        return handler;
    }

    public void info(Object message) {
        String m = String.valueOf(message);
        for (String s : MATCH_NEWLINE.split(m)) {
            logger.log(Level.INFO, s);
        }
    }

    public void severe(Object message) {
        String m = String.valueOf(message);
        for (String s : MATCH_NEWLINE.split(m)) {
            logger.log(Level.SEVERE, s);
        }
    }

    public void warning(Object message) {
        String m = String.valueOf(message);
        for (String s : MATCH_NEWLINE.split(m)) {
            logger.log(Level.WARNING, s);
        }
    }

    public SimpleDateFormat getDateFormat() {
        return dateFormat;
    }

    public void setDateFormat(SimpleDateFormat dateFormat) {
        this.dateFormat = dateFormat;
    }

    public Logger getLogger() {
        return logger;
    }

    public Object getSource() {
        return source;
    }
}
