package nl.lockhead.lpf.plugins;

import nl.lockhead.lpf.events.LPFEvent;
import nl.lockhead.lpf.exceptions.InvalidLPFEventException;
import nl.lockhead.lpf.plugins.annotations.PluginEvent;
import nl.lockhead.lpf.plugins.loaders.IPluginLoader;
import nl.lockhead.lpf.plugins.plugin.Plugin;
import nl.lockhead.lpf.plugins.plugin.PluginContainer;
import org.jetbrains.annotations.NotNull;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.*;
import java.util.concurrent.ConcurrentLinkedQueue;

public class PluginManager {

    private static PluginManager instance;
    private final ConcurrentLinkedQueue<PluginContainer> plugins;
    private final Map<PluginContainer, HashMap<Class<? extends LPFEvent>, Method>> registeredEvents =
            new HashMap<>();

    private PluginManager() {
        plugins = new ConcurrentLinkedQueue<>();
    }

    /**
     * @return the created PluginManager instance, or a new one if none have been created
     */
    public static PluginManager get() {
        return instance == null ? (instance = new PluginManager()) : instance;
    }

    @NotNull
    public static List<Method> getMethodsAnnotatedWith(final Class<?> type,
                                                       final Class<? extends Annotation> annotation) {
        final List<Method> methods = new ArrayList<>();
        Class<?> klass = type;
        while (klass != Object.class) {
            final List<Method> allMethods = new ArrayList<>(Arrays.asList(klass.getDeclaredMethods()));
            allMethods.stream().filter(method -> method.isAnnotationPresent(annotation)).forEach(methods::add);
            klass = klass.getSuperclass();
        }
        return methods;
    }

    public ConcurrentLinkedQueue<PluginContainer> getPlugins() {
        return plugins;
    }

    /**
     * Look through the plugin's associated methods, and look for the {@link PluginEvent} annotation.
     * Register those methods as listeners.
     *
     * @param plugin the plugin
     */
    @SuppressWarnings("unchecked")
    public void registerEvents(@NotNull Plugin plugin) {
        for (PluginContainer pc : plugins) {
            if (plugin.getId() == pc.getPlugin().getId()) {
                HashMap<Class<? extends LPFEvent>, Method> methods = new HashMap<>();
                for (Method method : getMethodsAnnotatedWith(plugin.getClass(), PluginEvent.class)) {
                    try {
                        Parameter[] p = method.getParameters();
                        if (p.length != 1) {
                            StringBuilder s = new StringBuilder();
                            for (Parameter par : p) {
                                s.append(par.getType().getSimpleName()).append(" ").append(par.getName()).append(", ");
                            }
                            if (s.length() > 0)
                                s = new StringBuilder(s.toString().trim().substring(0, s.length() - 2));
                            throw new InvalidLPFEventException(
                                    String.format("%s(%s) should only use one parameter directly extending superclass LPFEvent",
                                            method.getName(), s));
                        }
                        if (p[0].getType().getSuperclass() != LPFEvent.class) {
                            throw new InvalidLPFEventException(
                                    String.format("The parameter in %s(%s %s) should only directly extend superclass LPFEvent",
                                            method.getName(), p[0].getType().getSimpleName(), p[0].getName()));
                        }
                        methods.put((Class<? extends LPFEvent>) p[0].getType(), method);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                if (methods.size() > 0) {
                    registeredEvents.put(pc, methods);
                }
                break;
            }
        }
    }

    /**
     * Apply the given plugin loader to load plugins
     * @param loader the plugin loader to use
     */
    public void loadPlugins(@NotNull IPluginLoader loader) {
        plugins.addAll(loader.loadPlugins());
    }

    /**
     * Enable all loaded plugins.
     */
    public void enablePlugins() {
        for (PluginContainer pc : plugins) {
            try {
                pc.enablePlugin();
            } catch (UnsupportedOperationException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Disable all loaded plugins.
     */
    public void disablePlugins() {
        for (PluginContainer pc : plugins) {
            try {
                pc.disablePlugin();
            } catch (UnsupportedOperationException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Unload and remove all loaded plugins.
     */
    public void unloadPlugins() {
        for (PluginContainer pc : plugins) {
            try {
                pc.unloadPlugin();
            } catch (UnsupportedOperationException e) {
                e.printStackTrace();
            }
        }
        plugins.clear();
    }

    public final Map<PluginContainer, HashMap<Class<? extends LPFEvent>, Method>> getRegisteredEvents() {
        return registeredEvents;
    }
}
