/**
 * Copyright (C) 2010 Cloudfarming <info@cloudfarming.nl>
 *
 * Licensed under the Eclipse Public License - v 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.eclipse.org/legal/epl-v10.html
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nl.cloudfarming.client.area.field.explorer;

import java.awt.Dialog;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.persistence.RollbackException;
import javax.swing.JButton;
import javax.swing.JLabel;
import net.miginfocom.swing.MigLayout;
import nl.cloudfarming.client.area.AreaEventKey;
import nl.cloudfarming.client.area.field.AreaFieldModule;
import nl.cloudfarming.client.area.field.FieldAction;
import nl.cloudfarming.client.area.field.FieldAreaService;
import nl.cloudfarming.client.logging.AppLogFactory;
import nl.cloudfarming.client.logging.AppLogger;
import nl.cloudfarming.client.model.Field;
import nl.cloudfarming.eventbus.BaseGuiEventProducer;
import org.jdesktop.swingx.JXPanel;
import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;
import org.openide.util.lookup.ServiceProvider;

/**
 * Action which will remove a field from the application
 * @author Merijn Zengers
 */
@ServiceProvider(service = FieldAction.class)
public class RemoveFieldAction extends FieldAction {

    private static final String ACTION_NAME = NbBundle.getMessage(AreaFieldModule.class, "action.remove_field.name");
    private static final String BUTTON_OK = NbBundle.getMessage(AreaFieldModule.class, "action.remove_field.dialog.button.ok.text");
    private static final String BUTTON_CANCEL = NbBundle.getMessage(AreaFieldModule.class, "action.remove_field.dialog.button.cancel.text");
    private static final String DIALOG_DISPLAY_NAME = NbBundle.getMessage(AreaFieldModule.class, "action.remove_field.dialog.display_name");
    private static final String CASCADE_MESSAGE = NbBundle.getMessage(AreaFieldModule.class, "action.remove_field.dialog.cascade_error_message");
    private Dialog dlg;
    private final Field field;
    private FieldAreaService service;
    private final EventProducer eventProducer = new EventProducer();
    private static final AppLogger LOG = AppLogFactory.getLogger(AreaFieldModule.class);

    public RemoveFieldAction() {
        super(ACTION_NAME);
        field = Utilities.actionsGlobalContext().lookup(Field.class);
        service = getService();
    }

    @Override
    public void actionPerformed(ActionEvent e) {
        JXPanel panel = new JXPanel(new MigLayout("wrap 2"));

        String confirmationText = NbBundle.getMessage(AreaFieldModule.class, "action.remove_field.dialog.confirmation_text", field.getName());

        panel.add(new JLabel(confirmationText));
        // Create the dialog and add the ok and cancel buttons
        DialogDescriptor dd = new DialogDescriptor(panel, DIALOG_DISPLAY_NAME);

        // OK and Cancel-Buttons
        dd.setOptions(new Object[]{getOkButton(), getCancelButton()});

        dlg = DialogDisplayer.getDefault().createDialog(dd);
        dlg.setVisible(true);
    }

    /**
     * Creates the dialog ok button
     * @return The OK button
     */
    private JButton getOkButton() {
        JButton jButtonOK = new JButton();
        jButtonOK.setText(BUTTON_OK);
        jButtonOK.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent ae) {
                try {
                    service.removeField(field);
                } catch (RollbackException ex) {
                    LOG.info(CASCADE_MESSAGE);
                }
                eventProducer.triggerEvent(AreaEventKey.FIELD_REMOVED);
            }
        });
        return jButtonOK;
    }

    /**
     * creates the dialog cancel button
     * @return The cancel button
     */
    private JButton getCancelButton() {
        JButton jButtonCancel = new JButton();
        jButtonCancel.setText(BUTTON_CANCEL);
        jButtonCancel.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent ae) {
                dlg.setVisible(false);
                dlg = null;
            }
        });
        return jButtonCancel;
    }

    private FieldAreaService getService() {
        return service == null ? service = Lookup.getDefault().lookup(FieldAreaService.class) : service;
    }

    private class EventProducer extends BaseGuiEventProducer {

        public EventProducer() {
            super(AreaEventKey.class);
        }
    }

    public void setService(FieldAreaService service) {
        this.service = service;
    }
}
