/**
 * Copyright (C) 2010 Cloudfarming <info@cloudfarming.nl>
 *
 * Licensed under the Eclipse Public License - v 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.eclipse.org/legal/epl-v10.html
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nl.cloudfarming.client.area.field;

import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPolygon;
import java.beans.IntrospectionException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import nl.cloudfarming.client.area.AreaService;
import nl.cloudfarming.client.geoviewer.LayerObject;
import nl.cloudfarming.client.geoviewer.LayerObjectAction;
import nl.cloudfarming.client.model.HelpLine;
import nl.cloudfarming.client.util.GeometryUtil;
import org.openide.util.Lookup;

/**
 * Layerobject representing a plot
 * 
 * @author Timon Veenstra
 */
public class HelpLineLayerObject implements LayerObject<MultiLineString, HelpLine> {

    private final MultiLineString geometry;
    private final HelpLine helpLine;
    private AreaService service;
    private static final Logger LOG = Logger.getLogger("LineLayerObject.Actions");

    public HelpLineLayerObject(final HelpLine helpLine) throws IntrospectionException {
        this.helpLine = helpLine;
        geometry = (MultiLineString) GeometryUtil.wktToGeometry(helpLine.getWKT());
        service = getPlotAreaService();
    }

    @Override
    public MultiLineString getGeometry() {
        return this.geometry;
    }

    @Override
    public String getKey() {
        return Long.toString(this.helpLine.getId());
    }

    @Override
    public String getTooltipText() {
        return Long.toString(this.helpLine.getId()) + " " + this.helpLine.getName();
    }

    @Override
    public Long getKeyAttributeValue() {
        return this.helpLine.getId();
    }

    @Override
    public long getId() {
        return this.helpLine.getId();
    }

    @Override
    public HelpLine getObject() {
        return this.helpLine;
    }

            
    @Override
    public LayerObjectAction[] getActions() {
        List<LayerObjectAction> layerObjectActions = new ArrayList<LayerObjectAction>();
        for(LineAction lineAction : Lookup.getDefault().lookupAll(LineAction.class)){
            try{
                layerObjectActions.add(lineAction.getClass().newInstance());          
            } catch (InstantiationException ex) {
                LOG.log(Level.FINE, "Instantiation of new {0} failed", lineAction.getClass());
            } catch (IllegalAccessException ex) {
                LOG.log(Level.FINE, "Instantiation of new {0} failed", lineAction.getClass());
            }
        }
        return layerObjectActions.toArray(new LayerObjectAction[]{});
    }

// TODO: Save HelpLines
    @Override
    public void save() {
//        service.saveHelpLineChanges(this.helpLine);
    }

    protected final AreaService getPlotAreaService() {
        return (service == null) ? Lookup.getDefault().lookup(FieldAreaService.class) : service;
    }

    protected void setPlotAreaService(final FieldAreaService service) {
        this.service = service;
    }
    
}
