/**
 * Copyright (C) 2010 Cloudfarming <info@cloudfarming.nl>
 *
 * Licensed under the Eclipse Public License - v 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.eclipse.org/legal/epl-v10.html
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nl.cloudfarming.client.area.field;

import java.awt.BorderLayout;
import java.util.logging.Logger;
import org.openide.util.NbBundle;
import org.openide.windows.TopComponent;
import org.openide.windows.WindowManager;
import org.openide.util.ImageUtilities;
import org.netbeans.api.settings.ConvertAsProperties;
import org.openide.awt.UndoRedo;
import org.openide.explorer.ExplorerManager;
import org.openide.explorer.ExplorerUtils;
import org.openide.explorer.view.BeanTreeView;
import org.openide.util.Lookup;

/**
 * Top component which displays the field explorer.
 * @author Timon Veenstra
 */
@ConvertAsProperties(dtd = "-//nl.cloudfarming.client.area.field.explorer//FieldExplorer//EN",
autostore = false)
public final class FieldExplorerTopComponent extends TopComponent implements ExplorerManager.Provider {

    /**
     * Singleton instance of this FieldExplorerTopcomponent
     */
    private static FieldExplorerTopComponent instance;
    
    
    /**
     * ID to identify this FieldExplorerTopcomponent
     */
    private static final String PREFERRED_ID = "FieldExplorerTopComponent";
    
    /*
     * Persisting service for plots
     */
    private final FieldAreaService service;

    public FieldExplorerTopComponent() {
        initComponents();
        setName(NbBundle.getMessage(FieldExplorerTopComponent.class, "CTL_FieldExplorerTopComponent"));
        setToolTipText(NbBundle.getMessage(FieldExplorerTopComponent.class, "HINT_FieldExplorerTopComponent"));
        service = Lookup.getDefault().lookup(FieldAreaService.class);
        assert service != null;
        associateLookup(service.getLookup());

    }

    private void initComponents() {
        setLayout(new BorderLayout());
        add(new BeanTreeView(), BorderLayout.CENTER);
    }

    /**
     * Gets default instance. Do not use directly: reserved for *.settings files only,
     * i.e. deserialization routines; otherwise you could get a non-deserialized instance.
     * To obtain the singleton instance, use {@link #findInstance}.
     */
    public static synchronized FieldExplorerTopComponent getDefault() {
        if (instance == null) {
            instance = new FieldExplorerTopComponent();
        }
        return instance;
    }

    /**
     * Obtain the FieldExplorerTopComponent instance. Never call {@link #getDefault} directly!
     */
    public static synchronized FieldExplorerTopComponent findInstance() {
        TopComponent win = WindowManager.getDefault().findTopComponent(PREFERRED_ID);
        if (win == null) {
            Logger.getLogger(FieldExplorerTopComponent.class.getName()).warning(
                    "Cannot find " + PREFERRED_ID + " component. It will not be located properly in the window system.");
            return getDefault();
        }
        if (win instanceof FieldExplorerTopComponent) {
            return (FieldExplorerTopComponent) win;
        }
        Logger.getLogger(FieldExplorerTopComponent.class.getName()).warning(
                "There seem to be multiple components with the '" + PREFERRED_ID
                + "' ID. That is a potential source of errors and unexpected behavior.");
        return getDefault();
    }

    @Override
    public int getPersistenceType() {
        return TopComponent.PERSISTENCE_ALWAYS;
    }

    void writeProperties(final java.util.Properties p) {
        // better to version settings since initial version as advocated at
        // http://wiki.apidesign.org/wiki/PropertyFiles
        p.setProperty("version", "1.0");
    }

    Object readProperties(final java.util.Properties p) {
        if (instance == null) {
            instance = this;
        }
        instance.readPropertiesImpl(p);
        return instance;
    }

    private void readPropertiesImpl(final java.util.Properties p) {
        // String version = p.getProperty("version");
    }

    @Override
    protected String preferredID() {
        return PREFERRED_ID;
    }

   @Override
    public ExplorerManager getExplorerManager() {
        return service.getExplorerManager();
    }


    @Override
    public UndoRedo getUndoRedo() {
        return service.getUndoredoManager();
    }

    @Override
     public void addNotify() {
        super.addNotify();
        ExplorerUtils.activateActions(service.getExplorerManager(), true);
    }

    @Override
    public void removeNotify() {
        ExplorerUtils.activateActions(service.getExplorerManager(), false);
        super.removeNotify();
    }
}
