/**
 * Copyright (C) 2010 Cloudfarming <info@cloudfarming.nl>
 *
 * Licensed under the Eclipse Public License - v 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.eclipse.org/legal/epl-v10.html
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nl.cloudfarming.client.area.field;

import java.beans.IntrospectionException;
import java.util.Date;
import java.util.List;
import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.swing.ActionMap;
import javax.swing.text.DefaultEditorKit;
import nl.cloudfarming.client.area.AreaService;
import nl.cloudfarming.client.area.field.explorer.GeometryRootNode;
import nl.cloudfarming.client.model.AbLine;
import nl.cloudfarming.client.model.CommonModelService;
import nl.cloudfarming.client.model.HelpLine;
import nl.cloudfarming.client.model.PartField;
import nl.cloudfarming.client.model.Field;
import nl.cloudfarming.client.model.ShapeFile;
import nl.cloudfarming.client.model.ProductionUnit;
import org.openide.explorer.ExplorerManager;
import org.openide.explorer.ExplorerUtils;
import org.openide.util.Exceptions;
import org.openide.util.NbBundle;
import org.openide.util.lookup.ServiceProvider;

/**
 *
 * @author Timon Veenstra
 */
@ServiceProvider(service = FieldAreaService.class)
public class FieldAreaService extends CommonModelService implements AreaService {

    @Override
    protected String getModuleName() {
        return AreaFieldModule.MODULE_NAME;
    }

    final public List<Field> findAllCurrentFields() {
        startTransaction();
        Query query = getEntityManager().createNamedQuery(Field.Q_FIND_ALL_VALID_FIELDS);
        query.setParameter("date", new Date());
        return query.getResultList();
    }

    @Override
    final public void createNewField(final Field field) {
        persistInTransaction(field);
    }

    @Override
    public void createNewHelpLine(HelpLine helpLine) {
        persistInTransaction(helpLine);
    }

    @Override
    public void createNewAbLine(AbLine abLine) {
        persistInTransaction(abLine);
    }

    @Override
    final public void saveFieldChanges(final Field field) {
        mergeAndPersistInTransaction(field);
    }

    final public void savePartFieldChanges(final PartField partField) {
        mergeAndPersistInTransaction(partField);
    }

    @Override
    final public void createNewShapeFile(final ShapeFile shapeFile) {
        persistInTransaction(shapeFile);
    }

    /**
     * Creates a new partField
     * @param pf the partfield that should be created
     */
    public void createNewPartField(PartField pf) {
        persistInTransaction(pf);
    }

    /**
     * Find production units by date
     * @param date The PU should be active on this date 
     * @return The list of PU's active on the specified Date
     */
    @Override
    public List<ProductionUnit> findProductionUnitsByDate(Date date) {
        startTransaction();
        Query query = getEntityManager().createNamedQuery(ProductionUnit.Q_FIND_ALL_PRODUCTION_UNITS_BY_DATE);
        query.setParameter("date", date);
        return query.getResultList();
    }

    /**
     * Sets the endDate of the field to now.
     * @param field the field that should be ended
     * @retrun boolean true if succesfull false if not succesfull
     * @throws IllegalArgumentException if the Provided field is not a persisted entity
     */
    public boolean endField(Field field) {
        startTransaction();
        //First check if the field is a persisted entity
        if (field.getId() == 0) {
            throw new IllegalArgumentException("the provided field is not a persisted entity");
        }
        Date now = new Date();

        if (field.getStartDate() == null || field.getStartDate().before(now)) {
            field.setEndDate(now);
            saveFieldChanges(field);
            return true;
        }
        return false;
    }

    @Override
    protected void initExplorer() {
        try {
            ExplorerManager manager = new ExplorerManager();
            //manager.setRootContext(new FieldRootNode(new FieldLayer("service.explorer.layerlist.field_node.name", this)));

            FieldLayer fieldLayer = new FieldLayer("service.explorer.layerlist.field_node.name", this);
            HelpLineLayer lineLayer = new HelpLineLayer("service.explorer.layerlist.line_node.name", this);
            manager.setRootContext(new GeometryRootNode(fieldLayer, lineLayer));


            manager.getRootContext().setDisplayName(NbBundle.getMessage(this.getClass(), "service.explorer.rootnode.displayname"));

            ActionMap map = new ActionMap();
            map.put(DefaultEditorKit.copyAction, ExplorerUtils.actionCopy(manager));
            map.put(DefaultEditorKit.cutAction, ExplorerUtils.actionCut(manager));
            map.put(DefaultEditorKit.pasteAction, ExplorerUtils.actionPaste(manager));
            map.put("delete", ExplorerUtils.actionDelete(manager, true)); // or false

            setExplorerManager(manager, map);
        } catch (IntrospectionException ex) {
            Exceptions.printStackTrace(ex);
        }
    }

    /**
     * Find partfields by Date
     * @param date The date the field has to be active.
     * @return The list of partfields active on Date date
     */
    @Override
    public List<PartField> findPartFieldsByDate(Date date) {
        startTransaction();
        Query query = getEntityManager().createNamedQuery(PartField.Q_FIND_ALL_PARTFIELDS_BY_DATE);
        query.setParameter("date", date);
        return query.getResultList();
    }

    /**
     * Find a Field by specifying the filename and the checksum
     * @param name Filename
     * @param checksum file checksum
     * @return 
     */
    @Override
    public ShapeFile findShapeFileByNameAndChecksum(String name, String checksum) {
        startTransaction();
        Query query = getEntityManager().createNamedQuery(ShapeFile.Q_FIND_SHAPEFILE_BY_FILENAME_AND_CHECKSUM);
        query.setParameter("fileName", name);
        query.setParameter("checksum", checksum);
        try {
            return (ShapeFile) query.getSingleResult();
        } catch (NoResultException ner) {
            return null;
        }
    }

    /**
     * Find a Field by ID
     * @param id
     * @return 
     */
    @Override
    public Field findFieldById(long id) {
        startTransaction();
        return getEntityManager().find(Field.class, id);
    }

    /**
     * Find all the fields where the endDate is not filled
     * @return all the active Fields
     */
    public List<Field> findAllActiveFields() {
        startTransaction();
        Query query = getEntityManager().createNamedQuery(Field.Q_FIND_ALL_ACTIVE_FIELDS);
        return query.getResultList();
    }

    /**
     * Find all the fields where the endDate is not filled
     * @return all the active Fields
     */
    public List<HelpLine> findAllHelpLines() {
        startTransaction();
        Query query = getEntityManager().createNamedQuery(HelpLine.Q_FIND_ALL_HELP_LINES);
        return query.getResultList();
    }

    /**
     * Removes a field. If a field has partFields the field will not be removed. 
     * Instead the endDate of the field and all of its partFields will be filed.
     * @param field which will be removed
     */
    public void removeField(Field field) {
        startTransaction();
        if (field.getId() == 0) {
            throw new IllegalArgumentException("trying to remove non persisted field");
        }
        if (field.getPartFields().isEmpty()) {
            getEntityManager().remove(getEntityManager().merge(field));
            getEntityManager().getTransaction().commit();
        } else {
            endField(field);
        }
    }

    /**
     * Removes a line. If a field has partFields the field will not be removed. 
     * Instead the endDate of the field and all of its partFields will be filed.
     * @param field which will be removed
     */
    public void removeHelpLine(HelpLine line) {
        startTransaction();
        if (line.getId() == 0) {
            throw new IllegalArgumentException("trying to remove non persisted line");
        }
        getEntityManager().remove(getEntityManager().merge(line));
        getEntityManager().getTransaction().commit();
    }
}
