/**
 * Copyright (C) 2010 Cloudfarming <info@cloudfarming.nl>
 *
 * Licensed under the Eclipse Public License - v 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.eclipse.org/legal/epl-v10.html
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nl.cloudfarming.client.area.field;

import com.vividsolutions.jts.geom.MultiPolygon;
import java.beans.IntrospectionException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import nl.cloudfarming.client.area.AreaService;
import nl.cloudfarming.client.geoviewer.LayerObject;
import nl.cloudfarming.client.geoviewer.LayerObjectAction;
import nl.cloudfarming.client.model.Field;
import nl.cloudfarming.client.util.GeometryUtil;
import org.openide.util.Lookup;

/**
 * Layerobject representing a plot
 * 
 * @author Timon Veenstra
 */
public class FieldLayerObject implements LayerObject<MultiPolygon, Field> {

    private final MultiPolygon geometry;
    private final Field plot;
    private AreaService service;
    private static final Logger LOG = Logger.getLogger("FieldLayerObject.Actions");

    public FieldLayerObject(final Field plot) throws IntrospectionException {
        this.plot = plot;
        geometry = (MultiPolygon) GeometryUtil.wktToGeometry(plot.getWKT());
        service = getPlotAreaService();
    }

    @Override
    public MultiPolygon getGeometry() {
        return this.geometry;
    }

    @Override
    public String getKey() {
        return Long.toString(this.plot.getId());
    }

    @Override
    public String getTooltipText() {
        return Long.toString(this.plot.getId()) + " " + this.plot.getName();
    }

    @Override
    public Long getKeyAttributeValue() {
        return this.plot.getId();
    }

    @Override
    public Long getId() {
        return this.plot.getId();
    }

    @Override
    public Field getObject() {
        return this.plot;
    }

            
    @Override
    public LayerObjectAction[] getActions() {
        List<LayerObjectAction> layerObjectActions = new ArrayList<LayerObjectAction>();
        for(FieldAction fieldAction : Lookup.getDefault().lookupAll(FieldAction.class)){
            try{
                layerObjectActions.add(fieldAction.getClass().newInstance());          
            } catch (InstantiationException ex) {
                LOG.log(Level.FINE, "Instantiation of new {0} failed", fieldAction.getClass());
            } catch (IllegalAccessException ex) {
                LOG.log(Level.FINE, "Instantiation of new {0} failed", fieldAction.getClass());
            }
        }
        return layerObjectActions.toArray(new LayerObjectAction[]{});
    }

    @Override
    public void save() {
        service.saveFieldChanges(this.plot);
    }

    protected final AreaService getPlotAreaService() {
        return (service == null) ? Lookup.getDefault().lookup(FieldAreaService.class) : service;
    }

    protected void setPlotAreaService(final FieldAreaService service) {
        this.service = service;
    }
    
}
