package nl.bimbase.bimworks.client;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import nl.sascom.backplane.appbase.library.DefaultErrorCode;
import nl.sascom.backplanepublic.client.NodeClient;
import nl.sascom.backplanepublic.common.ClientTask;
import nl.sascom.backplanepublic.common.ExecuteException;
import nl.sascom.backplanepublic.common.LightContainerInterface;
import nl.sascom.backplanepublic.common.NodeTransport;
import nl.sascom.backplanepublic.common.Request;
import nl.sascom.backplanepublic.common.Response;
import nl.sascom.backplanepublic.common.StreamAlreadyRegisteredException;
import nl.sascom.backplanepublic.common.StreamManager;

/**
 * BimWorksClient object can be short-lived objects (but long-lived objects are fine too).
 * BimWorksClient objects are usually authenticated by either a username/password combination, a BIM.works API Token, or some other type of auth.
 * BimWorksClient objects cannot be constructed directly, but only through the BimWorksClientFactory.
 *
 */
public class BimWorksClient implements AutoCloseable {

	public static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

	static {
		OBJECT_MAPPER.findAndRegisterModules();
	}
	
	private final NodeClient nodeClient;
	
	public final FileSystemApi fs;
	public final GisApi gis;
	public final AuthApi auth;
	public final IfcApi ifc;
	public final UploadApi upload;
	public final TokensApi tokens;
	public final Bim bim;
	public final AdminApi admin;
	public final ModelChecks modelChecks;

	BimWorksClient(NodeClient nodeClient) {
		this.nodeClient = nodeClient;
		
		this.fs = new FileSystemApi(this);
		this.gis = new GisApi(this);
		this.auth = new AuthApi(this);
		this.ifc = new IfcApi(this);
		this.upload = new UploadApi(this);
		this.tokens = new TokensApi(this);
		this.bim = new Bim(this);
		this.admin = new AdminApi(this);
		this.modelChecks = new ModelChecks(this);
	}
	
	BimWorksClient(String connectionUrl) throws Exception {
		this(new NodeClient(connectionUrl));
	}

	BimWorksClient(NodeTransport nodeTransport) throws Exception {
		this(new NodeClient(nodeTransport));
	}

	BimWorksClient(NodeTransport nodeTransport, StreamManager streamManager) throws Exception {
		this(new NodeClient(nodeTransport, null, streamManager));
	}

	BimWorksClient(NodeTransport nodeTransport, LightContainerInterface nodeInterface) throws Exception {
		this(new NodeClient(nodeTransport, nodeInterface));
	}

	@Override
	public void close() throws InterruptedException {
		nodeClient.close();
	}

	public NodeClient getNodeClient() {
		return nodeClient;
	}
	
	public ObjectNode generateUploadToken(UUID parentUuid, Duration duration, Path path) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GenerateUploadModelToTreeToken");
		ObjectNode input = request.createObject();
		input.put("node_uuid", parentUuid.toString());
		input.put("expires_seconds", duration.getSeconds());
		input.put("filename", path.getFileName().toString());
		try {
			input.put("filesize", Files.size(path));
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		input.put("contentType", "application/ifc");
		request.setInput(input);
		try {
			Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
			return (ObjectNode) response.getOutput();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
		return null;
	}

	public UploadModelResponse uploadModelWithToken(Path path, ObjectNode payload) throws IOException, StreamAlreadyRegisteredException, NodeAlreadyExistsException, InterruptedException, BimWorksException {
		Request request = nodeClient.createRequest();
		request.setInput(payload.get("input"));
		request.setTask((ObjectNode)payload.get("task"));
		request.setTimeOut(4, TimeUnit.HOURS);
		String streamId = nodeClient.registerStream(com.google.common.io.Files.asByteSource(path.toFile()));
		request.attachStream(streamId);
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			return new UploadModelResponse((ObjectNode) task.await(4, TimeUnit.HOURS).getOutput());
		} catch (ExecuteException e) {
			if (e.getErrorCode().name().equals("DefaultErrorCode") && e.getErrorCode().getCode() == 29) {
				throw new NodeAlreadyExistsException(e.getUserSafeMessage());
			}
			throw new BimWorksException(e);
		}
	}

	public ObjectNode generateDownloadToken(UUID newNodeUuid, Duration duration) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GenerateDownloadToken");
		ObjectNode input = request.createObject();
		input.put("node_uuid", newNodeUuid.toString());
		input.put("expires_seconds", duration.getSeconds());
		request.setInput(input);
		try {
			Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
			return (ObjectNode) response.getOutput();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
		return null;
	}

	public ObjectNode generateQueryToken(Set<UUID> versionUuids, BimQuery bimQuery, String[] paths, Duration duration) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GenerateQueryToken");
		ObjectNode input = request.createObject();
		input.put("type", "JSON");
		input.put("output", "RESPONSE");
		ArrayNode versionUuidsNode = Response.createArray();
		for (UUID versionUuid : versionUuids) {
			versionUuidsNode.add(versionUuid.toString());
		}
		input.set("models", versionUuidsNode);
		input.set("query", bimQuery.toJson());
		ArrayNode pathsNode = Response.createArray();
		for (String path : paths) {
			pathsNode.add(path);
		}
		input.set("paths", pathsNode);
		input.put("expires_seconds", duration.getSeconds());
		request.setInput(input);
		try {
			Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
			return (ObjectNode) response.getOutput();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
		return null;
	}
	
	public String getApplicationVersion() throws IOException, ExecuteException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GetApplicationVersion");
		Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
		return response.getOutput().get("version").asText();
	}

	public ArrayNode listModelsForGis() throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setTaskName("ListModelsForGis");
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			return response.getArrayOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		return null;
	}

	public NodeClient getApi() {
		return this.nodeClient;
	}

	public String getLastScreenshotNew(UUID lastVersionUuid) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setTaskName("GetLastScreenshotNew");
		ArrayNode modelsNode = Response.createArray();
		modelsNode.add(lastVersionUuid.toString());
		request.getObjectInput().set("models", modelsNode);
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			if (response.hasStreams()) {
				
			}
			return null;
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		return null;
	}

	public JsonNode executeAsyncTask(Request request) throws BimWorksException {
		ClientTask task = this.nodeClient.createAsyncTask(request);
		try {
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			return response.getOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			throw new BimWorksException(DefaultErrorCode.INTERRUPTED);
		}
	}

	public Request createRequest() {
		return nodeClient.createRequest();
	}

	public ClientTask createAsyncTask(Request request) {
		return nodeClient.createAsyncTask(request);
	}

	public String registerStream(Path path) throws StreamAlreadyRegisteredException {
		return nodeClient.registerStream(path);
	}

	public String registerStream(String filename, String contentType, String url, long filesize) throws MalformedURLException, StreamAlreadyRegisteredException {
		return nodeClient.registerStream(filename, contentType, url, filesize);
	}

	public String registerStream(String filename, long filesize, String contentType, InputStream inputStream) throws StreamAlreadyRegisteredException {
		return nodeClient.registerStream(filename, filesize, contentType, inputStream);
	}

	public void setAuthToken(ObjectNode auth) {
		this.nodeClient.setAuth(auth);
		this.nodeClient.setAuth(auth);
		this.nodeClient.connectAsync(auth);
	}
	
	public ObjectNode executeForObject(Task taskInput) throws BimWorksException {
		try {
			ClientTask task = taskInput.createAsyncTask(this);
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			return response.getObjectOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			throw new BimWorksException(DefaultErrorCode.INTERRUPTED);
		}
	}

	public Response executeForResponse(Task taskInput) throws BimWorksException {
		try {
			ClientTask task = taskInput.createAsyncTask(this);
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			return response;
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			throw new BimWorksException(DefaultErrorCode.INTERRUPTED);
		}
	}
}