/**
 * Copyright (c) 2003-2004, Mark Lussier
 * All rights reserved.
 *
 * Portions Copyright (c) 2003 by David A. Czarnecki
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 * Neither the name of the "Mark Lussier" and "Sandler" nor the names of
 * its contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Products derived from this software may not be called "Sandler",
 * nor may "Sandler" appear in their name, without prior written permission of
 * Mark Lussier
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.intabulas.sandler.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * SandlerUtilities
 * 
 * @author Mark Lussier
 * @version $Id: SandlerUtilities.java,v 1.8 2004/02/05 01:30:40 intabulas Exp $
 */
public class SandlerUtilities {

    private SandlerUtilities() {
    }


    /**
     * RFC 822 style date format
     */
    public static final String RFC_822_DATE_FORMAT = "EEE, d MMM yyyy kk:mm:ss Z";

    /**
     * UTC style date format
     * 
     * @since blojsom 1.9.4
     */
    public static final String UTC_DATE_FORMAT = "yyyy-MM-dd'T'kk:mm:ss'Z'";


    /**
     * UTC format
     * SimpleDateFormats are not threadsafe, but we should not need more than one per
     * thread.
     */
    private static final ThreadLocal UTC_DATE_FORMAT_OBJECT = new ThreadLocal() {
        protected Object initialValue() {
            return new SimpleDateFormat(UTC_DATE_FORMAT);
        }
    };


    /**
     * Return a date in UTC style
     *
     * @param date Date
     * @return A String formatted as UTC
     */
    public static String getUTCDate(Date date) {
        String utcDate = ((SimpleDateFormat) UTC_DATE_FORMAT_OBJECT.get()).format(date).replaceAll("GMT", "");
        return utcDate;
    }


    /**
     * Return a date from an ISO 8601 style String
     * http://www.w3.org/TR/NOTE-datetime
     *
     * @param date String formatted as ISO 8601
     * @return Date
     */

    public static Date getDateFromUTCDate(String date) throws ParseException {
        return ((SimpleDateFormat) UTC_DATE_FORMAT_OBJECT.get()).parse(date);
    }


    /**
     * Returns the results of downloading the specified url
     *
     * @param url the URL to download
     * @return a String containing the contents of the response
     */
    public static String downloadUrl(String url) {
        String result = null;
        try {
            URL downloadUrl = new URL(url);
            HttpURLConnection http = (HttpURLConnection) downloadUrl.openConnection();
            http.setRequestProperty("User-Agent", "Sandler/0.5 ");
            BufferedInputStream in = new BufferedInputStream(http.getInputStream());
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            byte[] buffer = new byte[4096];
            for (; ;) {
                int n = in.read(buffer, 0, 4096);
                if (n < 0) {
                    break;
                }
                os.write(buffer, 0, n);
            }

            os.close();
            in.close();
            http.disconnect();
            result = os.toString();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return result;
    }


}
