/**
 * Copyright (c) 2003-2004, Mark Lussier
 * All rights reserved.
 *
 * Portions Copyright (c) 2003 by David A. Czarnecki
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 * Neither the name of the "Mark Lussier" and "Sandler" nor the names of
 * its contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Products derived from this software may not be called "Sandler",
 * nor may "Sandler" appear in their name, without prior written permission of
 * Mark Lussier
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.intabulas.sandler.elements.impl;

import org.intabulas.sandler.elements.*;
import org.intabulas.sandler.util.SandlerUtilities;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;
import java.text.MessageFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * FeedImpl
 * 
 * @author Mark Lussier
 * @version $Id: FeedImpl.java,v 1.11 2004/02/05 00:05:56 intabulas Exp $
 */
public class FeedImpl extends AbstractEntryElement implements Feed, AtomElement {

    private String _version = ATOM_VERSION;
    private String _language = "en";
    private Content _tagline;
    private Content _copyright;
    private Content _info;

    private Generator _generator;

    private List _entryList;


    public FeedImpl() {
        super();
        _entryList = new ArrayList(1);

    }


    public void clear() {
        _entryList.clear();
        _contributorList.clear();
        _version = ATOM_VERSION;
        _language = "en";
        _title = null;
        _linksList.clear();
        _modified = null;
        _tagline = null;
        _id = null;
        _copyright = null;
        _author = null;
        _generator = null;
    }


    // ===================================================

    /**
     * @param index 
     * @param entry 
     */
    public void addEntry(int index, Entry entry) {
        _entryList.add(index, entry);
    }

    /**
     * @param entry 
     */
    public boolean addEntry(Entry entry) {
        return _entryList.add(entry);
    }

    /**
     * @return 
     */
    public int getEntryCount() {
        return _entryList.size();
    }

    /**
     * @param index 
     */
    public void removeEntry(int index) {
        if (index >= 0 && index < _entryList.size()) {
            _entryList.remove(index);
        }
    }

    /**
     * @param entry
     * @return
     */
    public boolean removeEntry(Entry entry) {
        boolean result = false;
        if (_entryList.contains(entry)) {
            result = _entryList.remove(entry);
        }

        return result;
    }

    /**
     * @param index
     * @return
     */
    public Entry getEntry(int index) {
        Entry result = null;
        if (index >= 0 && index < _entryList.size()) {
            result = (Entry) _entryList.get(index);
        }
        return result;
    }



    // ===================================================


    /**
     * @param version 
     */
    public void setVersion(String version) {
        _version = version;
    }

    /**
     * @return 
     */
    public String getVersion() {
        return _version;
    }

    /**
     * @param language 
     */
    public void setLanguage(String language) {
        _language = language;
    }

    /**
     * @return 
     */
    public String getLanguage() {
        return _language;
    }

    /**
     * @param info 
     */
    public void setInfo(Content info) {
        _info = info;

    }

    /**
     * @param info 
     */
    public void setInfo(String info) {
        _info = new ContentImpl(ELEMENT_INFO);
        _info.setBody(info);

    }

    /**
     * @return 
     */
    public Content getInfo() {
        return _info;
    }

    /**
     * @return 
     */
    public Content getTagline() {
        return _tagline;
    }

    /**
     * @param tagline 
     */
    public void setTagline(Content tagline) {
        _tagline = tagline;
    }


    public void setTagline(String tagline) {
        _tagline = new ContentImpl(ELEMENT_TAGLINE);
        _tagline.setBody(tagline);
    }


    /**
     * @return 
     */
    public Content getCopyright() {
        return _copyright;
    }

    /**
     * @param copyright 
     */
    public void setCopyright(Content copyright) {
        _copyright = copyright;
    }


    /**
     * @param generator 
     */
    public void setGenerator(Generator generator) {
        _generator = generator;
    }

    /**
     * @return 
     */
    public Generator getGenerator() {
        return _generator;
    }

    /**
     * Returns a string representation of the object.
     * 
     * @return a String representation of the object.
     */
    public String toString() {

        StringBuffer buffer = new StringBuffer(XML_STARTDOC);
        String feedAttributes = MessageFormat.format(FORMAT_FEEDATTRIBUTES, new Object[]{_version, _language});
        buffer.append(MessageFormat.format(FORMAT_STARTELEMENT, new Object[]{ELEMENT_FEED, feedAttributes}));

        buffer.append(_title.toString());

        if (_linksList.size() > 0) {
            for (int x = 0; x < _linksList.size(); x++) {
                Link link = (Link) _linksList.get(x);
                buffer.append(link.toString());
            }

        }


        buffer.append(MessageFormat.format(FORMAT_STRINGELEMENT, new Object[]{ELEMENT_MODIFIED, "", SandlerUtilities.getUTCDate(_modified)}));

        if (_summary != null) {
            buffer.append(_summary.toString());
        }


        if (_tagline != null) {
            buffer.append(_tagline.toString());
        }

        if (_info != null) {
            buffer.append(_info.toString());
        }

        if (_id != null) {
            buffer.append(MessageFormat.format(FORMAT_STRINGELEMENT, new Object[]{ELEMENT_ID, "", _id}));
        }

        if (_copyright != null) {
            buffer.append(_copyright.toString());
        }


        if (_generator != null) {
            buffer.append(_generator.toString());
        }

        if (_author != null) {
            buffer.append(HTMLTAG_START).append(ELEMENT_AUTHOR).append(HTMLTAG_CLOSE);
            buffer.append(_author.toString());
            buffer.append(HTMLTAG_BEGIN).append(ELEMENT_AUTHOR).append(HTMLTAG_CLOSE);

        }


        if (_entryList.size() > 0) {
            for (int x = 0; x < _entryList.size(); x++) {
                Entry entry = (Entry) _entryList.get(x);
                buffer.append(entry.toString());
            }

        }

        if (_contributorList.size() > 0) {
            for (int x = 0; x < _contributorList.size(); x++) {
                Person contributor = (Person) _contributorList.get(x);
                buffer.append(HTMLTAG_START).append(ELEMENT_CONTRIBUTOR).append(HTMLTAG_CLOSE);
                buffer.append(contributor.toString());
                buffer.append(HTMLTAG_BEGIN).append(ELEMENT_CONTRIBUTOR).append(HTMLTAG_CLOSE);

            }

        }


        buffer.append(HTMLTAG_BEGIN).append(ELEMENT_FEED).append(HTMLTAG_CLOSE);


        return buffer.toString();

    }


    public void loadDocument(XmlPullParser parser) throws XmlPullParserException {


        int eventType = parser.getEventType();

        do {
//            if (eventType == XmlPullParser.START_DOCUMENT) {
//            }
            if (eventType == XmlPullParser.START_TAG) {
                if (parser.getName().equalsIgnoreCase(ELEMENT_FEED)) {
                    processDocumentAttributes(parser);
                    try {
                        processDocumenteElements(parser);
                    } catch (XmlPullParserException e) {
                        e.printStackTrace();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
//            if (eventType == XmlPullParser.END_TAG) {
//            }
            try {
                eventType = parser.next();
            } catch (XmlPullParserException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        } while (eventType != XmlPullParser.END_DOCUMENT);
    }


    private void processDocumenteElements(XmlPullParser parser) throws XmlPullParserException, IOException {

        String boundingTag = parser.getName();
        String childTag = null;
        int eventType = parser.nextTag();

        do {
            if (eventType == XmlPullParser.START_TAG) {
                childTag = parser.getName();

                if (parser.getName().equalsIgnoreCase(ELEMENT_TITLE)) {
                    processTitleTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_LINK)) {
                    LinkImpl link = new LinkImpl();
                    link.loadDocument(parser);
                    _linksList.add(link);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_ID)) {
                    processIdTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_MODIFIED)) {
                    processModifiedTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_TAGLINE)) {
                    processTaglineTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_INFO)) {
                    processInfoTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_COPYRIGHT)) {
                    processCopyrightTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_GENERATOR)) {
                    _generator = new GeneratorImpl();
                    _generator.loadDocument(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_ENTRY)) {
                    Entry entry = new EntryImpl();
                    entry.loadDocument(parser);
                    addEntry(entry);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_AUTHOR)) {
                    _author = new PersonImpl();
                    _author.loadDocument(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_CONTRIBUTOR)) {
                    PersonImpl contrib = new PersonImpl();
                    contrib.loadDocument(parser);
                    _contributorList.add(contrib);
                }

            }
//            if (eventType == XmlPullParser.END_TAG) {
//            }
            try {
                eventType = parser.next();
            } catch (XmlPullParserException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        } while (!(eventType == XmlPullParser.END_TAG && ELEMENT_FEED.equals(parser.getName())));


    }


    private void processDocumentAttributes(XmlPullParser parser) {
        int attributeCount = parser.getAttributeCount();
        for (int x = 0; x < attributeCount; x++) {
            if (parser.getAttributeName(x).equals(ATTRIBUTE_VERSION)) {
                _version = parser.getAttributeValue(x);
            }
            if (parser.getAttributeName(x).equals(ATTRIBUTE_LANG_NP)) {
                _language = parser.getAttributeValue(x);
            }

        }
    }


    private void processTitleTag(XmlPullParser parser) throws XmlPullParserException, IOException {
        Content content = new ContentImpl(ELEMENT_TITLE);
        content.loadDocument(parser);
        setTitle(content);
    }

    private void processCopyrightTag(XmlPullParser parser) throws XmlPullParserException, IOException {
        Content content = new ContentImpl(ELEMENT_COPYRIGHT);
        content.loadDocument(parser);
        setCopyright(content);
    }


    private void processIdTag(XmlPullParser parser) throws XmlPullParserException, IOException {
        _id = parser.nextText();
    }

    private void processTaglineTag(XmlPullParser parser) throws XmlPullParserException, IOException {
        Content content = new ContentImpl(ELEMENT_TAGLINE);
        content.loadDocument(parser);
        setTagline(content);
    }


    private void processInfoTag(XmlPullParser parser) throws XmlPullParserException, IOException {
        Content content = new ContentImpl(ELEMENT_INFO);
        content.loadDocument(parser);
        setInfo(content);
    }

    private void processModifiedTag(XmlPullParser parser) throws XmlPullParserException, IOException {
        try {
            _modified = SandlerUtilities.getDateFromUTCDate(parser.nextText());
        } catch (ParseException e) {
            _modified = new Date();
        }

    }


}
