/**
 * Copyright (c) 2003-2004, Mark Lussier
 * All rights reserved.
 *
 * Portions Copyright (c) 2003 by David A. Czarnecki
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 * Neither the name of the "Mark Lussier" and "Sandler" nor the names of
 * its contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Products derived from this software may not be called "Sandler",
 * nor may "Sandler" appear in their name, without prior written permission of
 * Mark Lussier
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.intabulas.sandler.elements.impl;

import org.intabulas.sandler.elements.*;
import org.intabulas.sandler.util.SandlerUtilities;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;
import java.text.MessageFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * EntryImpl
 * 
 * @author Mark Lussier
 * @version $Id: EntryImpl.java,v 1.10 2004/02/05 00:05:56 intabulas Exp $
 */
public class EntryImpl extends AbstractEntryElement implements Entry, AtomElement {

    private Date _created;
    private Date _issued;
    private List _contentList;


    public EntryImpl() {
        super();
        _contentList = new ArrayList(1);
    }


    public Content getSummary() {
        return _summary;
    }

    public void setSummary(Content summary) {
        _summary = summary;
    }

    /**
     * @return 
     */
    public Date getCreated() {
        return _created;
    }

    /**
     * @param date 
     */
    public void setIssued(Date date) {
        _issued = date;
    }

    /**
     * @param date 
     */
    public void setCreated(Date date) {
        _created = date;
    }

    /**
     * @return 
     */
    public Date getIssued() {
        return _issued;
    }


    /**
     * @param content 
     */
    public boolean addContent(Content content) {
        return _contentList.add(content);
    }

    /**
     * @return 
     */
    public int getContentCount() {
        return _contentList.size();
    }

    /**
     * @param index
     * @return
     */
    public Content getContent(int index) {
        Content result = null;
        if (index >= 0 && index < _contentList.size()) {
            result = (Content) _contentList.get(index);
        }
        return result;
    }

    /**
     * @param content
     * @return
     */
    public boolean removeContent(Content content) {
        return _contentList.remove(content);
    }

    /**
     * @param index 
     */
    public void removeContent(int index) {
        if (index >= 0 && index < _contentList.size()) {
            _contentList.remove(index);
        }
    }

    /**
     * @param index   
     * @param content 
     */
    public void addContent(int index, Content content) {
        _contentList.add(index, content);
    }

    /**
     * Returns a string representation of the object.
     * 
     * @return a String representation of the object.
     */
    public String toString() {

        //@todo Add XMLNS: Entry when marshalled alone

        StringBuffer buffer = new StringBuffer();
        buffer.append(MessageFormat.format(FORMAT_STARTELEMENT, new Object[]{ELEMENT_ENTRY, "", ""})); //" " + ATTRIBUTE_XMLNS + '=' + ATOM_NAMESPACE}));

        buffer.append(_title.toString());
        buffer.append(MessageFormat.format(FORMAT_STRINGELEMENT, new Object[]{ELEMENT_ID, "", _id}));

        if (_summary != null) {
            buffer.append(_summary.toString());
        }


        if (_linksList.size() > 0) {
            for (int x = 0; x < _linksList.size(); x++) {
                Link link = (Link) _linksList.get(x);
                buffer.append(link.toString());
            }

        }

        if (_modified != null) {
            buffer.append(MessageFormat.format(FORMAT_STRINGELEMENT, new Object[]{ELEMENT_MODIFIED, "", SandlerUtilities.getUTCDate(_modified)}));
        }
        if (_issued != null) {
            buffer.append(MessageFormat.format(FORMAT_STRINGELEMENT, new Object[]{ELEMENT_ISSUED, "", SandlerUtilities.getUTCDate(_issued)}));
        }

        if (_created != null) {
            buffer.append(MessageFormat.format(FORMAT_STRINGELEMENT, new Object[]{ELEMENT_CREATED, "", SandlerUtilities.getUTCDate(_created)}));
        }
        if (_summary != null) {
            buffer.append(MessageFormat.format(FORMAT_STRINGELEMENT, new Object[]{ELEMENT_SUMMARY, "", _summary}));
        }

        if (_author != null) {
            buffer.append(HTMLTAG_START).append(ELEMENT_AUTHOR).append(HTMLTAG_CLOSE);
            buffer.append(_author.toString());
            buffer.append(HTMLTAG_BEGIN).append(ELEMENT_AUTHOR).append(HTMLTAG_CLOSE);
        }


        if (_contributorList.size() > 0) {
            for (int x = 0; x < _contributorList.size(); x++) {
                Person contributor = (Person) _contributorList.get(x);
                buffer.append(HTMLTAG_START).append(ELEMENT_CONTRIBUTOR).append(HTMLTAG_CLOSE);
                buffer.append(contributor.toString());
                buffer.append(HTMLTAG_BEGIN).append(ELEMENT_CONTRIBUTOR).append(HTMLTAG_CLOSE);

            }

        }

        if (_contentList.size() > 0) {
            for (int x = 0; x < _contentList.size(); x++) {
                Content content = (Content) _contentList.get(x);
                if (content != null) buffer.append(content.toString());
            }

        }
        buffer.append(HTMLTAG_BEGIN).append(ELEMENT_ENTRY).append(HTMLTAG_CLOSE);


        return buffer.toString();

    }

    public void loadDocument(XmlPullParser parser) throws XmlPullParserException {
        int eventType = parser.getEventType();
        do {
            if (eventType == XmlPullParser.START_TAG) {

                if (parser.getName().equalsIgnoreCase(ELEMENT_TITLE)) {
                    Content content = new ContentImpl(ELEMENT_TITLE);
                    content.loadDocument(parser);
                    setTitle(content);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_LINK)) {
                    LinkImpl link = new LinkImpl();
                    link.loadDocument(parser);
                    _linksList.add(link);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_ID)) {
                    try {
                        _id = parser.nextText();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_SUMMARY)) {
                    Content content = new ContentImpl(ELEMENT_SUMMARY);
                    content.loadDocument(parser);
                    setSummary(content);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_MODIFIED)) {
                    processModifiedTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_ISSUED)) {
                    processIssuedTag(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_CREATED)) {
                    processCreatedTag(parser);

                } else if (parser.getName().equalsIgnoreCase(ELEMENT_AUTHOR)) {
                    _author = new PersonImpl();
                    _author.loadDocument(parser);
                } else if (parser.getName().equalsIgnoreCase(ELEMENT_CONTENT)) {
                    Content content = new ContentImpl();
                    content.loadDocument(parser);
                    _contentList.add(content);

                } else if (parser.getName().equalsIgnoreCase(ELEMENT_CONTRIBUTOR)) {
                    PersonImpl contrib = new PersonImpl();
                    contrib.loadDocument(parser);
                    _contributorList.add(contrib);
                }

//            } else if (eventType == XmlPullParser.END_TAG) {
            }
            try {
                eventType = parser.next();
            } catch (XmlPullParserException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }

        } while (!(eventType == XmlPullParser.END_TAG && ELEMENT_ENTRY.equals(parser.getName())));


    }

    private void processModifiedTag(XmlPullParser parser) throws XmlPullParserException {
        try {
            _modified = SandlerUtilities.getDateFromUTCDate(parser.nextText());
        } catch (ParseException e) {
            _modified = new Date();
        } catch (IOException e) {
            _modified = null;
        }

    }

    private void processCreatedTag(XmlPullParser parser) throws XmlPullParserException {
        try {
            _created = SandlerUtilities.getDateFromUTCDate(parser.nextText());
        } catch (ParseException e) {
            _created = new Date();
        } catch (IOException e) {
            _created = null;
        }

    }

    private void processIssuedTag(XmlPullParser parser) throws XmlPullParserException {
        try {
            _issued = SandlerUtilities.getDateFromUTCDate(parser.nextText());
        } catch (ParseException e) {
            _issued = new Date();
        } catch (IOException e) {
            _issued = null;
        }

    }


}
