/**
 * Copyright (c) 2003-2004, Mark Lussier
 * All rights reserved.
 *
 * Portions Copyright (c) 2003 by David A. Czarnecki
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 * Neither the name of the "Mark Lussier" and "Sandler" nor the names of
 * its contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Products derived from this software may not be called "Sandler",
 * nor may "Sandler" appear in their name, without prior written permission of
 * Mark Lussier
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.intabulas.sandler.elements.impl;

import org.intabulas.sandler.elements.AtomElement;
import org.intabulas.sandler.elements.Content;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;


/**
 * ContentImpl
 * 
 * @author Mark Lussier
 * @version $Id: ContentImpl.java,v 1.7 2004/01/02 01:43:57 intabulas Exp $
 */
public class ContentImpl implements Content, AtomElement {

    private String _mimeType;
    private String _language;
    private String _mode;
    private String _content;
    private String _tagname = ELEMENT_CONTENT;

    public ContentImpl() {
        this(null, ELEMENT_CONTENT);

    }

    public ContentImpl(String tagname) {
        this(null, tagname);
    }

    public ContentImpl(String content, String tagname) {
        _tagname = tagname;
        _content = content;
    }


    /**
     * Returns the MIME Type format of the body content
     * 
     * @return a String representing the MIME Type
     */
    public String getMimeType() {
        return _mimeType;
    }

    /**
     * Set's the MIME Type format of the body content
     * 
     * @param mimeType a String representing the MIME Type
     */
    public void setMimeType(String mimeType) {
        _mimeType = mimeType;
    }


    /**
     * Returns the method used to encode the body content (ie: xml, escaped, base64)
     * 
     * @return a String representing the encoding
     */
    public String getMode() {
        return _mode;
    }

    /**
     * Set's the method used to encode the body content (ie: xml, escaped, base64)
     * 
     * @param mode a String representing the encoding
     */
    public void setMode(String mode) {
        _mode = mode;
    }


    /**
     * Retutns the language of the body content (xml:lang)
     * 
     * @return a String representing the content language
     */
    public String getLanguage() {
        return _language;
    }

    /**
     * Sets the language of the body content (xml:lang)
     * 
     * @param language a String representing the content language
     */
    public void setLanguage(String language) {
        _language = language;
    }


    /**
     * Returns the actual entry content
     * 
     * @return content a String representing the body content
     */
    public String getBody() {
        return _content;
    }

    /**
     * Set's the entry content
     * 
     * @param content a String representing the body content
     */
    public void setBody(String content) {
        _content = content;
    }


    /**
     * Set's the content tag name
     * 
     * @param tagame 
     */
    public void setTagName(String tagame) {
        _tagname = tagame;
    }

    /**
     * Returns a string representation of the object.
     * 
     * @return a String representation of the object.
     */
    public String toString() {


        StringBuffer buffer = new StringBuffer();
        buffer.append(HTMLTAG_START).append(_tagname);
        if (_mimeType != null) {
            buffer.append(SPACE).append(ATTRIBUTE_TYPE).append("=\"").append(_mimeType).append("\"");
        }
        if (_mode != null) {
            buffer.append(SPACE).append(ATTRIBUTE_MODE).append("=\"").append(_mode).append("\"");
        }
        if (_language != null) {
            buffer.append(SPACE).append(ATTRIBUTE_LANG).append("=\"").append(_language).append("\"");
        }
        buffer.append(HTMLTAG_CLOSE);
        if (_tagname.equals(ELEMENT_CONTENT)) {
            buffer.append(CDATA_START).append(_content).append(CDATA_END);
        } else {
            buffer.append(_content);
        }
        buffer.append(HTMLTAG_BEGIN).append(_tagname).append(HTMLTAG_CLOSE);


        return buffer.toString();

    }

    /**
     * Extract relevant content from the parser stream
     * 
     * @param parser the XML Pull Parser instabce
     * 
     * @throws XmlPullParserException 
     */
    public void loadDocument(XmlPullParser parser) throws XmlPullParserException {
        int eventType = parser.getEventType();
        StringBuffer buffer = new StringBuffer();

        do {
            if (eventType == XmlPullParser.START_TAG && _tagname.equals(parser.getName())) {
                processDocumentAttributes(parser);
            } else {
                buffer.append(parser.getText());
            }
            try {
                eventType = parser.next();
            } catch (XmlPullParserException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        } while (!(eventType == XmlPullParser.END_TAG && _tagname.equals(parser.getName())));

        _content = buffer.toString();

    }

    private void processDocumentAttributes(XmlPullParser parser) {
        int attributeCount = parser.getAttributeCount();
        for (int x = 0; x < attributeCount; x++) {
            if (parser.getAttributeName(x).equals(ATTRIBUTE_LANG_NP)) {
                setLanguage(parser.getAttributeValue(x));
            } else if (parser.getAttributeName(x).equals(ATTRIBUTE_MODE)) {
                setMode(parser.getAttributeValue(x));
            } else if (parser.getAttributeName(x).equals(ATTRIBUTE_TYPE)) {
                setMimeType(parser.getAttributeValue(x));
            }


        }
    }


}
