/**
 * Copyright (c) 2003-2004, Mark Lussier
 * All rights reserved.
 *
 * Portions Copyright (c) 2003 by David A. Czarnecki
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 * Neither the name of the "Mark Lussier" and "Sandler" nor the names of
 * its contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Products derived from this software may not be called "Sandler",
 * nor may "Sandler" appear in their name, without prior written permission of
 * Mark Lussier
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.intabulas.sandler;

import org.intabulas.sandler.elements.*;
import org.intabulas.sandler.elements.impl.*;
import org.intabulas.sandler.exceptions.FeedMarshallException;

import java.util.Date;

/**
 * SyndicationFactory
 * 
 * @author lussiema
 * @version: $Id: SyndicationFactory.java,v 1.7 2004/02/05 03:08:25 frums Exp $
 */
public class SyndicationFactory implements AtomElement {

    /**
     * Allow override
     */
    public SyndicationFactory() {
    }


    /**
     * Create a new/empty Atom Feed instance
     * 
     * @return a Feed instance
     */
    public static Feed newSyndicationFeed() {
        Feed result = new FeedImpl();
        return result;
    }

    /**
     * Create a new/empty Atom Feed instance from a String representation of a Feed
     *
     * @param feed a String containing a Atom Feed document
     * @return a Feed instance
     * @throws FeedMarshallException
     */
    public static Feed newSyndicationFeed(String feed) throws FeedMarshallException {
        Feed result = Sandler.unmarshallFeed(feed);
        return result;
    }


    /**
     * Create a new/empty Atom Entry instance
     * 
     * @return an Entry instance
     */
    public static Entry newSyndicationEntry() {
        Entry result = new EntryImpl();
        result.setCreated(new Date());
        return result;
    }

    /**
     * Create a new/empty Atom Entry instance from a String representation of an Entry
     *
     * @param entry a String containinh a Atom Entry document
     * @return an Entry instance
     * @throws FeedMarshallException
     */
    public static Entry newSyndicationEntry(String entry) throws FeedMarshallException {
        Entry result = Sandler.unmarshallEntry(entry);
        result.setCreated(new Date());
        return result;
    }

    /**
     * Create new instance of an Person type
     */
    public static Person createPerson(String name, String email, String url) {
        PersonImpl person = new PersonImpl();
        person.setEmail(email);
        person.setName(name);
        person.setUrl(url);
        return person;
    }

    /**
     * Create text/plain escaped Content
     */
    public static Content createEscapedContent(String body) {
        ContentImpl content = new ContentImpl();
        content.setMimeType(AtomConstants.Type.TEXT_HTML);
        content.setBody(body);
        content.setMode(AtomConstants.Mode.ESCAPED);
        return content;
    }

    /**
     * Create text/xml Content (does not validate!)
     */
    public static Content createTextXmlContent(String body) {
        ContentImpl content = new ContentImpl();
        content.setMimeType(AtomConstants.Type.TEXT_XML);
        content.setBody(body);
        content.setMode(AtomConstants.Mode.XML);
        return content;
    }

    /**
     * Convenience method for building the oft-needed "alternate" link
     */
    public static Link createAlternateLink(String href, String title) {
        LinkImpl link = new LinkImpl();
        link.setHref(href);
        link.setRelationship(AtomConstants.Rel.ALTERNATE);
        link.setType(AtomConstants.Type.TEXT_HTML);
        if (title != null) link.setTitle(title);
        return link;
    }

    /**
     * Convenience method for building the oft-needed "alternate" link
     */
    public static Link createAlternateLink(String href) {
        return createAlternateLink(href, null);
    }

    /**
     * Construct a valid Atom Feed
     */
    public static Feed createFeed(Person author, String title, Link alternate, Date modified) {
        FeedImpl feed = new FeedImpl();
        feed.setAuthor(author);
        feed.setVersion(AtomConstants.VERSION);
        feed.setLanguage(AtomConstants.Lang.EN_US);
        feed.setTitle(title);
        feed.addLink(alternate);
        feed.setModified(modified);
        return feed;
    }

    /**
     * Construct a valid Atom Entry without any content
     */
    public static Entry createEntry(String title, String id, Link alternate, Date issued, Date modified) {
        return createEntry(null, title, id, alternate, issued, modified);
    }

    /**
     * Construct a valid Atom Entry with some content
     */
    public static Entry createEntry(Content content, String title, String id, Link alternate, Date issued, Date modified) {
        EntryImpl entry = new EntryImpl();
        entry.setTitle(title);
        entry.addLink(alternate);
        entry.setModified(modified);
        entry.setIssued(issued);
        entry.addContent(content);
        entry.setId(id);

        return entry;
    }

}
