/*
 * Decompiled with CFR 0.152.
 */
package net.shibboleth.utilities.java.support.security;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyException;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import javax.annotation.Nonnull;
import javax.crypto.SecretKey;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.codec.Base64Support;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resource.Resource;
import net.shibboleth.utilities.java.support.security.DataExpiredException;
import net.shibboleth.utilities.java.support.security.DataSealerException;
import org.bouncycastle.crypto.BlockCipher;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.engines.AESEngine;
import org.bouncycastle.crypto.modes.GCMBlockCipher;
import org.bouncycastle.crypto.params.AEADParameters;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataSealer
extends AbstractInitializableComponent {
    @Nonnull
    private Logger log = LoggerFactory.getLogger(DataSealer.class);
    @NonnullAfterInit
    private SecretKey cipherKey;
    @NonnullAfterInit
    private SecureRandom random;
    @NonnullAfterInit
    private String keystoreType = "JCEKS";
    @NonnullAfterInit
    private Resource keystoreResource;
    @NonnullAfterInit
    private String keystorePassword;
    @NonnullAfterInit
    private String cipherKeyAlias;
    @NonnullAfterInit
    private String cipherKeyPassword;

    @Override
    public void doInitialize() throws ComponentInitializationException {
        try {
            try {
                Constraint.isNotNull(this.keystoreType, "Keystore type cannot be null");
                Constraint.isNotNull(this.keystoreResource, "Keystore resource cannot be null");
                Constraint.isNotNull(this.keystorePassword, "Keystore password cannot be null");
                Constraint.isNotNull(this.cipherKeyAlias, "Cipher key alias cannot be null");
                Constraint.isNotNull(this.cipherKeyPassword, "Cipher key password cannot be null");
            }
            catch (ConstraintViolationException e) {
                throw new ComponentInitializationException(e);
            }
            if (this.random == null) {
                this.random = new SecureRandom();
            }
            this.cipherKey = this.loadKey(this.cipherKeyAlias);
            this.testEncryption();
        }
        catch (IOException | GeneralSecurityException e) {
            this.log.error(e.getMessage());
            throw new ComponentInitializationException("Exception loading the keystore", e);
        }
        catch (DataSealerException e) {
            this.log.error(e.getMessage());
            throw new ComponentInitializationException("Exception testing the encryption settings used", e);
        }
    }

    @NonnullAfterInit
    public SecretKey getCipherKey() {
        return this.cipherKey;
    }

    @NonnullAfterInit
    public SecureRandom getRandom() {
        return this.random;
    }

    @NonnullAfterInit
    public String getKeystoreType() {
        return this.keystoreType;
    }

    @NonnullAfterInit
    public Resource getKeystoreResource() {
        return this.keystoreResource;
    }

    @NonnullAfterInit
    public String getKeystorePassword() {
        return this.keystorePassword;
    }

    @NonnullAfterInit
    public String getCipherKeyAlias() {
        return this.cipherKeyAlias;
    }

    @NonnullAfterInit
    public String getCipherKeyPassword() {
        return this.cipherKeyPassword;
    }

    public void setRandom(@Nonnull SecureRandom r) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        this.random = Constraint.isNotNull(r, "SecureRandom cannot be null");
    }

    public void setKeystoreType(@Nonnull @NotEmpty String type) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        this.keystoreType = Constraint.isNotNull(StringSupport.trimOrNull(type), "Keystore type cannot be null or empty");
    }

    public void setKeystoreResource(@Nonnull @NotEmpty Resource resource) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        this.keystoreResource = Constraint.isNotNull(resource, "Keystore resource cannot be null");
    }

    public void setKeystorePassword(@Nonnull @NotEmpty String password) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        this.keystorePassword = Constraint.isNotNull(password, "Keystore password cannot be null");
    }

    public void setCipherKeyAlias(@Nonnull @NotEmpty String alias) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        this.cipherKeyAlias = Constraint.isNotNull(StringSupport.trimOrNull(alias), "Cipher key alias cannot be null or empty");
    }

    public void setCipherKeyPassword(@Nonnull @NotEmpty String password) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        this.cipherKeyPassword = Constraint.isNotNull(password, "Cipher key password cannot be null");
    }

    @Nonnull
    public String unwrap(@Nonnull @NotEmpty String wrapped) throws DataSealerException {
        try {
            SecretKey keyUsed;
            byte[] in = Base64Support.decode(wrapped);
            ByteArrayInputStream inputByteStream = new ByteArrayInputStream(in);
            DataInputStream inputDataStream = new DataInputStream(inputByteStream);
            String keyAlias = inputDataStream.readUTF();
            this.log.trace("Data was encrypted by key '{}'", (Object)keyAlias);
            if (keyAlias.equals(this.cipherKeyAlias)) {
                keyUsed = this.cipherKey;
            } else {
                keyUsed = this.loadKey(keyAlias);
                this.log.trace("Loaded older key '{}' from keystore", (Object)keyAlias);
            }
            GCMBlockCipher cipher = new GCMBlockCipher((BlockCipher)new AESEngine());
            int ivSize = cipher.getUnderlyingCipher().getBlockSize();
            byte[] iv = new byte[ivSize];
            inputDataStream.readFully(iv);
            AEADParameters aeadParams = new AEADParameters(new KeyParameter(keyUsed.getEncoded()), 128, iv, keyAlias.getBytes());
            cipher.init(false, (CipherParameters)aeadParams);
            byte[] data = new byte[in.length - ivSize];
            int dataSize = inputDataStream.read(data);
            byte[] plaintext = new byte[cipher.getOutputSize(dataSize)];
            int outputLen = cipher.processBytes(data, 0, dataSize, plaintext, 0);
            cipher.doFinal(plaintext, outputLen);
            return this.extractAndCheckDecryptedData(plaintext);
        }
        catch (IOException | IllegalStateException | InvalidCipherTextException e) {
            this.log.error(e.getMessage());
            throw new DataSealerException("Exception unwrapping data", (Exception)e);
        }
        catch (GeneralSecurityException e) {
            this.log.error(e.getMessage());
            throw new DataSealerException("Exception loading legacy key", e);
        }
    }

    @Nonnull
    private String extractAndCheckDecryptedData(@Nonnull @NotEmpty byte[] decryptedBytes) throws DataSealerException {
        try {
            ByteArrayInputStream byteStream = new ByteArrayInputStream(decryptedBytes);
            GZIPInputStream compressedData = new GZIPInputStream(byteStream);
            DataInputStream dataInputStream = new DataInputStream(compressedData);
            long decodedExpirationTime = dataInputStream.readLong();
            String decodedData = dataInputStream.readUTF();
            if (System.currentTimeMillis() > decodedExpirationTime) {
                this.log.info("Unwrapped data has expired");
                throw new DataExpiredException("Unwrapped data has expired");
            }
            this.log.debug("Unwrapped data verified");
            return decodedData;
        }
        catch (IOException e) {
            this.log.error(e.getMessage());
            throw new DataSealerException("Caught IOException unwrapping data", e);
        }
    }

    @Nonnull
    public String wrap(@Nonnull @NotEmpty String data, long exp) throws DataSealerException {
        if (data == null || data.length() == 0) {
            throw new IllegalArgumentException("Data must be supplied for the wrapping operation");
        }
        try {
            GCMBlockCipher cipher = new GCMBlockCipher((BlockCipher)new AESEngine());
            byte[] iv = new byte[cipher.getUnderlyingCipher().getBlockSize()];
            this.random.nextBytes(iv);
            AEADParameters aeadParams = new AEADParameters(new KeyParameter(this.cipherKey.getEncoded()), 128, iv, this.cipherKeyAlias.getBytes());
            cipher.init(true, (CipherParameters)aeadParams);
            ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
            GZIPOutputStream compressedStream = new GZIPOutputStream(byteStream);
            DataOutputStream dataStream = new DataOutputStream(compressedStream);
            dataStream.writeLong(exp);
            dataStream.writeUTF(data);
            dataStream.flush();
            compressedStream.flush();
            compressedStream.finish();
            byteStream.flush();
            byte[] plaintext = byteStream.toByteArray();
            byte[] encryptedData = new byte[cipher.getOutputSize(plaintext.length)];
            int outputLen = cipher.processBytes(plaintext, 0, plaintext.length, encryptedData, 0);
            outputLen += cipher.doFinal(encryptedData, outputLen);
            ByteArrayOutputStream finalByteStream = new ByteArrayOutputStream();
            DataOutputStream finalDataStream = new DataOutputStream(finalByteStream);
            finalDataStream.writeUTF(this.cipherKeyAlias);
            finalDataStream.write(iv);
            finalDataStream.write(encryptedData, 0, outputLen);
            finalDataStream.flush();
            finalByteStream.flush();
            return Base64Support.encode(finalByteStream.toByteArray(), false);
        }
        catch (IOException | IllegalStateException | InvalidCipherTextException e) {
            this.log.error(e.getMessage());
            throw new DataSealerException("Exception wrapping data", (Exception)e);
        }
    }

    private void testEncryption() throws DataSealerException {
        String decrypted;
        try {
            GCMBlockCipher cipher = new GCMBlockCipher((BlockCipher)new AESEngine());
            byte[] iv = new byte[cipher.getUnderlyingCipher().getBlockSize()];
            this.random.nextBytes(iv);
            AEADParameters aeadParams = new AEADParameters(new KeyParameter(this.cipherKey.getEncoded()), 128, iv, "aad".getBytes(StandardCharsets.UTF_8));
            cipher.init(true, (CipherParameters)aeadParams);
            byte[] plaintext = "test".getBytes(StandardCharsets.UTF_8);
            byte[] encryptedData = new byte[cipher.getOutputSize(plaintext.length)];
            int outputLen = cipher.processBytes(plaintext, 0, plaintext.length, encryptedData, 0);
            cipher.doFinal(encryptedData, outputLen);
            cipher.init(false, (CipherParameters)aeadParams);
            plaintext = new byte[cipher.getOutputSize(encryptedData.length)];
            outputLen = cipher.processBytes(encryptedData, 0, encryptedData.length, plaintext, 0);
            cipher.doFinal(plaintext, outputLen);
            decrypted = Strings.fromUTF8ByteArray((byte[])plaintext);
        }
        catch (IllegalStateException | InvalidCipherTextException e) {
            this.log.error("Round trip encryption/decryption test unsuccessful", e);
            throw new DataSealerException("Round trip encryption/decryption test unsuccessful", (Exception)e);
        }
        if (decrypted == null || !"test".equals(decrypted)) {
            this.log.error("Round trip encryption/decryption test unsuccessful. Decrypted text did not match");
            throw new DataSealerException("Round trip encryption/decryption test unsuccessful");
        }
    }

    @Nonnull
    private SecretKey loadKey(@Nonnull @NotEmpty String alias) throws GeneralSecurityException, IOException {
        KeyStore ks = KeyStore.getInstance(this.keystoreType);
        ks.load(this.keystoreResource.getInputStream(), this.keystorePassword.toCharArray());
        Key loadedKey = ks.getKey(alias, this.cipherKeyPassword.toCharArray());
        if (loadedKey == null) {
            this.log.error("Key '{}' not found", (Object)alias);
            throw new KeyException("Key was not found in keystore");
        }
        if (!(loadedKey instanceof SecretKey)) {
            this.log.error("Key '{}' is not a symmetric key", (Object)alias);
            throw new KeyException("Key was of incorrect type");
        }
        return (SecretKey)loadedKey;
    }
}

