/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.collection;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.NonnullFunction;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.NonnullSupplier;

/**
 * Support functions for Collection and Map Management.
 */
public final class CollectionSupport {
    
    /** logger. */
    @Nonnull private static Logger log = LoggerFactory.getLogger(CollectionSupport.class);
    
    /** Constructor. */
    private CollectionSupport() {
    }

    /** Build something we can plug in into {@link 
     * Collectors#toMap(java.util.function.Function, java.util.function.Function, BinaryOperator)}.
     * @param <T> the type we'll be looking at
     * @param what What we are building (i.e. IdpUI) 
     * @param takeFirst do we want the first of the last to win.
     * @return an appropriate {@link BinaryOperator}
     */
    @Nonnull public static <T> BinaryOperator<T> warningMergeFunction(final String what, final boolean takeFirst) {
        
        return new BinaryOperator<>() {

            public T apply(final T current, final T lookingAt) {
                log.warn("Duplicate detected building {}", what);
                log.debug("Values provided are {} and {} taking {}", current, lookingAt,
                        takeFirst ?"first":"last");
                return takeFirst? current:lookingAt ;
            }
            
        };        
    }

    /**
     * Return a collector which collects to a {@link NonnullSupplier} whose
     * content is the value returned by the collector you were given.
     *
     * @param <T> the type of input elements to the reduction operation of the collector
     * @param <A> the mutable accumulation type of the reduction operation of the collector (often
     *            hidden as an implementation detail)
     * @param <R> the result type of the reduction operation of the collector
     * @param collector the collector
     * @return the {@link NonnullSupplier}.
     */
    public static <T,A,R> Collector<T, A, NonnullSupplier<R>> nonnullCollector(final Collector<T,A,R> collector) {
        final Function<R, NonnullSupplier<R>> func =
                new NonnullFunction<R, NonnullSupplier<R>>() {
                    @Override @Nonnull
                    public NonnullSupplier<R> apply(@Nullable final R input) {
                        return NonnullSupplier.of(Constraint.isNotNull(input, "Null result from collector"));
                    }
                };
        return Collectors.collectingAndThen(collector, func);
    }

    /**
     * Gets an empty list with non-null guarantee.
     *
     * @param <T> list type
     *
     * @return empty list
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> emptyList() {
        return Collections.emptyList();
    }

    /**
     * Gets a singleton list with non-null guarantee.
     *
     * @param <T> list type
     * @param item the single item
     * 
     * @return singleton list
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> singletonList(@Nonnull final T item) {
        return Collections.singletonList(item);
    }

    /**
     * Gets a zero member list with non-null guarantee.
     * 
     * @param <T> list type
     *
     * @return zero member list
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> listOf() {
        return List.of();
    }

    /**
     * Gets a one member list with non-null guarantee.
     * 
     * @param <T> list type
     * @param first the first item
     *
     * @return a singleton list
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> listOf(@Nonnull final T first) {
        return List.of(first);
    }

    /**
     * Gets a two member list with non-null guarantee.
     * 
     * @param <T> list type
     * @param first the first item
     * @param second the second item
     *
     * @return two member list
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> listOf(@Nonnull final T first, @Nonnull final T second) {
        return List.of(first, second);
    }

    /**
     * Gets a list with non-null guarantee and a variable number of members.
     *
     * @param <T> list type
     * @param elements the elements
     *
     * @return multi member list
     */
    @SafeVarargs
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> listOf(@Nonnull final T... elements) {
        return List.of(elements);
    }

    /**
     * Copies a collection to a list with non-null guarantee.
     *
     * @param <T> list type
     *
     * @param coll collection to copy
     *
     * @return immutable copied list
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> copyToList(@Nonnull final Collection<? extends T> coll) {
        return List.copyOf(coll);
    }

    /**
     * Gets an empty set with non-null guarantee.
     *
     * @param <T> set type
     *
     * @return empty set
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> Set<T> emptySet() {
        return Collections.emptySet();
    }

    /**
     * Gets a singleton set with non-null guarantee.
     *
     * @param <T> set type
     * @param item the single item
     *
     * @return singleton set
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> Set<T> singleton(@Nonnull final T item) {
        return Collections.singleton(item);
    }

    /**
     * Gets a zero member set with non-null guarantee.
     *
     * @param <T> list type
     *
     * @return zero member set
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> Set<T> setOf() {
        return Set.of();
    }

    /**
     * Gets a one member set with non-null guarantee.
     *
     * @param <T> list type
     * @param first the first item
     *
     * @return one member set
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> Set<T> setOf(@Nonnull final T first) {
        return Set.of(first);
    }

    /**
     * Gets a two member set with non-null guarantee.
     *
     * @param <T> list type
     * @param first the first item
     * @param second the second item
     *
     * @return two member set
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> Set<T> setOf(@Nonnull final T first, @Nonnull final T second) {
        return Set.of(first, second);
    }

    /**
     * Gets a set with non-null guarantee and a variable number of members.
     *
     * @param <T> list type
     * @param elements the elements
     *
     * @return multi member set
     */
    @SafeVarargs
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> Set<T> setOf(@Nonnull final T... elements) {
        return Set.of(elements);
    }

    /**
     * Copies a collection to a set with non-null guarantee.
     * 
     * @param <T> set type
     * 
     * @param coll collection to copy
     * 
     * @return immutable copied set
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> Set<T> copyToSet(@Nonnull final Collection<? extends T> coll) {
        return Set.copyOf(coll);
    }
    
    /**
     * Gets an empty map with non-null guarantee.
     * 
     * @param <T> key type
     * @param <U> value type
     * 
     * @return empty amp
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T,U> Map<T,U> emptyMap() {
        return Collections.emptyMap();
    }
    
    /**
     * Gets a singleton map with non-null guarantee.
     * 
     * @param <T> key type
     * @param <U> value type
     * 
     * @param key map key
     * @param value map value
     * 
     * @return singleton amp
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T,U> Map<T,U> singletonMap(@Nonnull final T key,
            @Nullable final U value) {
        return Collections.singletonMap(key, value);
    }

    /**
     * Copies a map to a map with non-null guarantee.
     * 
     * @param <U> key type
     * @param <T> value type
     * 
     * @param map map to copy
     * 
     * @return immutable copied map
     */
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <U,T> Map<U,T> copyToMap(
            @Nonnull final Map<? extends U, ? extends T> map) {
        return Map.copyOf(map);
    }
 
    /** Creates a mutable List from an array.
     * @param <T> the class of the objects in the array
     * @param a the array by which the list will be backed
     * @return a list view of the specified array
     * @throws NullPointerException if the specified array is {@code null}
     */
    @SafeVarargs
    @SuppressWarnings("null")
    @Nonnull @Unmodifiable @NotLive public static <T> List<T> arrayAsList(@Nonnull final T... a) {
        return Arrays.asList(a);
    }
}