/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security;

import java.security.KeyException;

import javax.annotation.Nonnull;
import javax.crypto.SecretKey;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.Pair;
import net.shibboleth.shared.logic.Constraint;

/**
 * Interface for acquiring secret keys on behalf of the {@link DataSealer} class,
 * this principally abstracts methods of handling key versioning moreso than actual
 * access to keys, by optimizing access to keys in accordance with the manner in which
 * key rollover is handled.
 * 
 * <p>Implementations are expected to guarantee good performance for access to a current
 * "default" key, and may offer less performance on access to non-default keys.</p>
 */
public interface DataSealerKeyStrategy {

    /**
     * Get the default/current key to use for new operations, returned along with an identifier for it.
     * 
     * @return  the key
     * @throws KeyException if the key cannot be returned
     * 
     * @deprecated
     */
    @Deprecated(since="9.0.0", forRemoval=true)
    @Nonnull Pair<String,SecretKey> getDefaultKey() throws KeyException;
    
    /**
     * Get an immutable record of the default named key.
     * 
     * @return default key record
     * 
     * @throws KeyException if the key is unobtainable
     */
    @Nonnull default NamedKey getDefaultKeyRecord() throws KeyException {
        final Pair<String,SecretKey> result = getDefaultKey();
        return new NamedKey(result.getFirst(), result.getSecond());
    }
    
    /**
     * Get a specifically named key.
     * 
     * @param name name of the key to retrieve
     * 
     * @return  the key
     * @throws KeyException if the key cannot be returned, does not exist, etc.
     */
    @Nonnull SecretKey getKey(@Nonnull @NotEmpty final String name) throws KeyException;
 
    /**
     * Encapsulates a named key managed by a strategy.
     * 
     * @param name key name 
     * @param key key value
     * 
     * @since 9.0.0
     */
    record NamedKey(@Nonnull @NotEmpty String name, @Nonnull SecretKey key) {

        /**
         * Constructor.
         */
        public NamedKey {
            Constraint.isTrue(name != null && name.length() > 0, "Name cannot be empty or null");
            Constraint.isNotNull(key, "Key cannot be null");
        }
        
    }
    
}