/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.servlet.HttpServletRequestValidator;

/**
 * Component that validates HTTP request parameters for required presence, uniqueness and mutual exclusivity.
 */
public class BasicHttpServletRequestParametersValidator extends AbstractInitializableComponent
        implements HttpServletRequestValidator {
    
    /** Logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(BasicHttpServletRequestParametersValidator.class);
    
    /** Flag indicating whether to enforce the allowed parameters. */
    private boolean enforceAllowedParameters = false;
    
    /** Allowed parameters. */
    @Nonnull private Set<String> allowedParameters = CollectionSupport.emptySet();
    
    /** Required parameters. */
    @Nonnull private Set<String> requiredParameters = CollectionSupport.emptySet();
    
    /** Unique parameters. */
    @Nonnull private Set<String> uniqueParameters = CollectionSupport.emptySet();
    
    /** Mutually exclusive parameters. */
    @Nonnull private Set<Set<String>> mutuallyExclusiveParameters = CollectionSupport.emptySet();

    /**
     * Get flag indicating whether to enforce the allowed parameters. 
     * 
     * <p>Defaults to <code>false</code>.
     * 
     * @return flag indicating enforcement of allowed parameters
     */
    public boolean isEnforceAllowedParameters() {
        return enforceAllowedParameters;
    }

    /**
     * Set flag indicating whether to enforce the allowed parameters. 
     * 
     * <p>Defaults to <code>false</code>.
     * 
     * @param flag indicating enforcement of allowed parameters
     */
    public void setEnforceAllowedParameters(final boolean flag) {
        checkSetterPreconditions();
        enforceAllowedParameters = flag;
    }


    /**
     * Get the allowed parameters. 
     * 
     * <p>When enforced, a request parameter not in the allowed set will be considered invalid.</p>
     * 
     * @return the allowed parameters
     */
    @Nonnull @Unmodifiable @NotLive public Set<String> getAllowedParameters() {
        return allowedParameters;
    }

    /**
     * Set the allowed parameters. 
     * 
     * <p>When enforced, a request parameter not in the allowed set will be considered invalid.</p>
     * 
     * @param params the allowed parameters
     */
    public void setAllowedParameters(@Nullable final Set<String> params) {
        checkSetterPreconditions();
        allowedParameters = CollectionSupport.copyToSet(StringSupport.normalizeStringCollection(params));
    }

    /**
     * Get the required parameters. 
     * 
     * <p>A required parameter must be present in the request.</p>
     * 
     * @return the required parameters
     */
    @Nonnull @Unmodifiable @NotLive public Set<String> getRequiredParameters() {
        return requiredParameters;
    }

    /**
     * Set the required parameters. 
     * 
     * <p>A required parameter must be present in the request.</p>
     * 
     * @param params the required parameters
     */
    public void setRequiredParameters(@Nullable final Set<String> params) {
        checkSetterPreconditions();
        requiredParameters = CollectionSupport.copyToSet(StringSupport.normalizeStringCollection(params));
    }

    /**
     * Get the unique parameters. 
     * 
     * <p>A unique parameter must have at most 1 value.</p>
     * 
     * @return the unique parameters
     */
    @Nonnull @Unmodifiable @NotLive public Set<String> getUniqueParameters() {
        return uniqueParameters;
    }

    /**
     * Set the unique parameters. 
     * 
     * <p>A unique parameter must have at most 1 value.</p>
     * 
     * @param params the unique parameters
     */
    public void setUniqueParameters(@Nullable final Set<String> params) {
        checkSetterPreconditions();
        uniqueParameters = CollectionSupport.copyToSet(StringSupport.normalizeStringCollection(params));
    }

    /**
     * Get the mutually exclusive parameters.
     * 
     * <p>A request may not contain more then 1 parameter from an exclusivity set (the "inner" set(s) configured).
     * Multiple exclusivity sets may be specified.</p>
     * 
     * @return the mutually exclusive parameters
     */
    @Nonnull @Unmodifiable @NotLive public Set<Set<String>> getMutuallyExclusiveParameters() {
        return mutuallyExclusiveParameters;
    }

    /**
     * Set the mutually exclusive parameters.
     * 
     * <p>A request may not contain more then 1 parameter from an exclusivity set (the "inner" set(s) configured).
     * Multiple exclusivity sets may be specified.</p>
     * 
     * @param params the mutually exclusive parameters 
     */
    public void setMutuallyExclusiveParameters(@Nullable final Set<Set<String>> params) {
        checkSetterPreconditions();
        if (params == null) {
            mutuallyExclusiveParameters = CollectionSupport.emptySet();
        } else {
            mutuallyExclusiveParameters = params.stream()
                    .map(StringSupport::normalizeStringCollection)
                    .map(CollectionSupport::copyToSet)
                    .collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableSet())).get();
        }
    }

    /** {@inheritDoc} */
    public void validate(@Nonnull final HttpServletRequest request) throws ServletException {
        Constraint.isNotNull(request, "HttpServletRequest was null");
        final Set<String> requestParams = request.getParameterMap().keySet();
        
        if (isEnforceAllowedParameters()) {
            log.debug("Evaluating request for allowed parameters: {}", getAllowedParameters());
            for (final String requestParam : requestParams) {
                if (!getAllowedParameters().contains(requestParam)) {
                    log.warn("HTTP request contained a disallowed parameter: {}", requestParam);
                    throw new ServletException("HTTP request contained a disallowed parameter: " + requestParam);
                }
            }
        } else {
            log.debug("Enforcement of allowed parameters is disabled");
        }

        log.debug("Evaluating request for required parameters: {}", getRequiredParameters());
        for (final String param : getRequiredParameters()) {
            if (!requestParams.contains(param)) {
                log.warn("HTTP request did not contain required parameter: {}", param);
                throw new ServletException("HTTP request did not contain required parameter: " + param);
            }
        }
        
        log.debug("Evaluating request for unique parameters: {}", getUniqueParameters());
        for (final String param : getUniqueParameters()) {
            final String[] values = request.getParameterValues(param);
            if (values != null && values.length > 1) {
                log.warn("HTTP request contained {} values for parameter: {}", values.length, param);
                throw new ServletException("HTTP request contained multiple values for parameter: " + param);
            }
        }
        
        log.debug("Evaluating request for mutually exclusive parameters: {}", getMutuallyExclusiveParameters());
        for (final Set<String> group : getMutuallyExclusiveParameters())  {
            if (group.size() < 2) {
                log.debug("Exclusivity group had < 2 members, skipping evaluation: ", group);
                continue;
            }
            
            final Set<String> groupIntersection = requestParams.stream()
                    .filter(p -> group.contains(p))
                    .collect(Collectors.toSet());

           if (groupIntersection.size() > 1) {
               log.warn("HTTP request contained mutuallly exclusive parameters: {}", groupIntersection);
               throw new ServletException("HTTP request contained mutually exclusive parameters: "
                       + groupIntersection);
           }
       }
        
    }

}
