/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.httpclient;

import java.io.IOException;

import javax.annotation.Nonnull;

import org.apache.hc.core5.http.EntityDetails;
import org.apache.hc.core5.http.HeaderElements;
import org.apache.hc.core5.http.HttpException;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.HttpRequest;
import org.apache.hc.core5.http.HttpRequestInterceptor;
import org.apache.hc.core5.http.HttpVersion;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.slf4j.Logger;

import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Adds a <code>Connection: close</code> to all HTTP/1.1 requests.
 * 
 * <p>
 * This interceptor essentially disables connection keep-alive support and, by virtue of the server closing the
 * connection, prevents a {@link org.apache.hc.client5.http.io.HttpClientConnectionManager} from holding open and
 * reusing connections. If you'd like to allow the {@link org.apache.hc.client5.http.io.HttpClientConnectionManager}
 * to hold open connections for a while and potentially reuse them, consider using 
 * {@link HttpClientBuilder#setEvictIdleConnections(boolean)} and 
 * {@link HttpClientBuilder#setConnectionMaxIdleTime(java.time.Duration)} to limit the amount of time
 * the connections are held open.
 * </p>
 */
public class RequestConnectionClose implements HttpRequestInterceptor {
    
    /** Logger. */
    @Nonnull static final Logger LOG = LoggerFactory.getLogger(RequestConnectionClose.class);

    /** {@inheritDoc} */
    public void process(final HttpRequest request, final EntityDetails entity, final HttpContext context)
            throws HttpException, IOException {
        LOG.trace("Saw negotiated protocol version: {}", context.getProtocolVersion());
        if (HttpVersion.HTTP_1_1.lessEquals(context.getProtocolVersion())) {
            LOG.trace("Adding request header '{}: {}'", HttpHeaders.CONNECTION, HeaderElements.CLOSE);
            request.addHeader(HttpHeaders.CONNECTION, HeaderElements.CLOSE);
        } else {
            LOG.warn("Negotiated protocol did not match condition, skipping add of request header '{}: {}', "
                    + "may cause issues related to pooling", HttpHeaders.CONNECTION, HeaderElements.CLOSE);
        }
    }

}