/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.primitive.DeprecationSupport;
import net.shibboleth.shared.primitive.DeprecationSupport.ObjectType;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Legacy stub for compatibility.
 */
public class StubbedFilter implements Filter {
    
    /** Class name to warn about, defaults to the name of this class (or its superclass). */
    @Nonnull @NotEmpty private final String className;
    
    /** Constructor. */
    public StubbedFilter() {
        this(null);
    }
    
    /**
     * Constructor.
     *
     * @param name overrides class name for warning messages
     */
    public StubbedFilter(@Nullable @NotEmpty @ParameterName(name="name") final String name) {
        String s = StringSupport.trimOrNull(name);
        if (s == null) {
            s = "Servlet Filter " + getClass().getName();
        }
        className = s;
    }
    
    /** {@inheritDoc} */
    @Override
    public void init(final FilterConfig filterConfig) throws ServletException {
        DeprecationSupport.warn(ObjectType.CONFIGURATION, className, "edit-webapp/WEB-INF/web.xml", null);
    }

    /** {@inheritDoc} */
    @Override
    public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain)
            throws IOException, ServletException {
        DeprecationSupport.warnOnce(ObjectType.CONFIGURATION, className, "edit-webapp/WEB-INF/web.xml", null);
        chain.doFilter(request, response);
    }

}