/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.httpclient;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.ClientProtocolException;
import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.routing.RoutingSupport;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.HttpException;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.io.HttpClientResponseHandler;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.apache.hc.core5.io.ModalCloseable;
import org.slf4j.Logger;

import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Basic abstract implementation of {@link HttpClient}.
 * 
 * Based on {@link org.apache.hc.client5.http.impl.classic.CloseableHttpClient}.
 */
public abstract class AbstractHttpClient implements HttpClient, ModalCloseable {
    
    /** Logger. */
    private static final Logger LOG = LoggerFactory.getLogger(AbstractHttpClient.class);

    /**
     * Execute the HTTP request.
     * 
     * @param target the target host for the request, may be {@code null}
     * @param request the request to execute
     * @param context the context to use for the execution, or {@code null} to use the default context
     * @return the response to the request
     * @throws IOException in case of a problem or the connection was aborted
     */
    protected abstract ClassicHttpResponse doExecute(
            @Nullable final HttpHost target,
            @Nonnull final ClassicHttpRequest request,
            @Nullable final HttpContext context) throws IOException;

    /**
     * Determine the {@link HttpHost} instance based on the specified request.
     * 
     * @param request the request being executed
     * @return the {@link HttpHost} instance reflecting the specified request
     * @throws ClientProtocolException if target host can not be determined
     */
    private static HttpHost determineTarget(@Nullable final ClassicHttpRequest request) throws ClientProtocolException {
        try {
            return RoutingSupport.determineHost(request);
        } catch (final HttpException ex) {
            throw new ClientProtocolException(ex);
        }
    }

    /** {@inheritDoc} */
    public ClassicHttpResponse execute(
            @Nullable final HttpHost target,
            @Nullable final ClassicHttpRequest request,
            @Nullable final HttpContext context) throws IOException {
        return doExecute(target, Constraint.isNotNull(request, "HTTP request"), context);
    }

    /** {@inheritDoc} */
    public ClassicHttpResponse execute(
            @Nullable final ClassicHttpRequest request,
            @Nullable final HttpContext context) throws IOException {
        return doExecute(determineTarget(request), Constraint.isNotNull(request, "HTTP request"), context);
    }

    /** {@inheritDoc} */
    public ClassicHttpResponse execute(
            @Nullable final ClassicHttpRequest request) throws IOException {
        return doExecute(determineTarget(request), Constraint.isNotNull(request, "HTTP request"), null);
    }

    /** {@inheritDoc} */
    public ClassicHttpResponse execute(
            @Nullable final HttpHost target,
            @Nullable final ClassicHttpRequest request) throws IOException {
        return doExecute(target, Constraint.isNotNull(request, "HTTP request"), null);
    }

    /** {@inheritDoc} */
    public <T> T execute(
            @Nullable final ClassicHttpRequest request,
            @Nullable final HttpClientResponseHandler<? extends T> responseHandler) throws IOException {
        Constraint.isNotNull(request, "HTTP request");
        Constraint.isNotNull(responseHandler, "HTTP response handler");
        return execute(request, null, responseHandler);
    }

    /** {@inheritDoc} */
    public <T> T execute(
            @Nullable final ClassicHttpRequest request,
            @Nullable final HttpContext context,
            @Nullable final HttpClientResponseHandler<? extends T> responseHandler) throws IOException {
        Constraint.isNotNull(request, "HTTP request");
        Constraint.isNotNull(responseHandler, "HTTP response handler");
        final HttpHost target = determineTarget(request);
        return execute(target, request, context, responseHandler);
    }

    /** {@inheritDoc} */
    public <T> T execute(
            @Nullable final HttpHost target,
            @Nullable final ClassicHttpRequest request,
            @Nullable final HttpClientResponseHandler<? extends T> responseHandler) throws IOException {
        Constraint.isNotNull(request, "HTTP request");
        Constraint.isNotNull(responseHandler, "HTTP response handler");
        return execute(target, request, null, responseHandler);
    }

    /** {@inheritDoc} */
    public <T> T execute(
            @Nullable final HttpHost target,
            @Nullable final ClassicHttpRequest request,
            @Nullable final HttpContext context,
            @Nullable final HttpClientResponseHandler<? extends T> responseHandler) throws IOException {
        final HttpClientResponseHandler<? extends T> checkedHandler =
                Constraint.isNotNull(responseHandler, "HTTP response handler");

        try (final ClassicHttpResponse response =
                doExecute(target, Constraint.isNotNull(request, "HTTP request"), context)) {
            try {
                final T result = checkedHandler.handleResponse(response);
                final HttpEntity entity = response.getEntity();
                EntityUtils.consume(entity);
                return result;
            } catch (final HttpException t) {
                // Try to salvage the underlying connection in case of a protocol exception
                final HttpEntity entity = response.getEntity();
                try {
                    EntityUtils.consume(entity);
                } catch (final Exception t2) {
                    // Log this exception. The original exception is more
                    // important and will be thrown to the caller.
                    LOG.warn("Error consuming content after an exception.", t2);
                }
                throw new ClientProtocolException(t);
            }
        }
    }

}
