/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.saml.xmlobject.impl;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.saml.common.AbstractSAMLObjectUnmarshaller;
import org.w3c.dom.Attr;

import com.google.common.base.Strings;

import net.shibboleth.oidc.saml.xmlobject.DefaultAcrValue;
import net.shibboleth.oidc.saml.xmlobject.OAuthRPExtensions;
import net.shibboleth.oidc.saml.xmlobject.PostLogoutRedirectUri;
import net.shibboleth.oidc.saml.xmlobject.RequestUri;

/**
 * Unmarshaller for {@link OAuthRPExtensions} objects.
 */
public class OAuthRPExtensionsUnmarshaller extends AbstractSAMLObjectUnmarshaller {

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    protected void processChildElement(final XMLObject parentSAMLObject, final XMLObject childSAMLObject)
            throws UnmarshallingException {
        final OAuthRPExtensions descriptor = (OAuthRPExtensions) parentSAMLObject;

        if (childSAMLObject instanceof DefaultAcrValue) {
            descriptor.getDefaultAcrValues().add((DefaultAcrValue) childSAMLObject);
        } else if (childSAMLObject instanceof RequestUri) {
            descriptor.getRequestUris().add((RequestUri) childSAMLObject);
        } else if (childSAMLObject instanceof PostLogoutRedirectUri) {
            descriptor.getPostLogoutRedirectUris().add((PostLogoutRedirectUri) childSAMLObject);
        } else {
            descriptor.getUnknownXMLObjects().add(childSAMLObject);
        }
    }
    
    /** {@inheritDoc} */
    protected void processAttribute(final XMLObject samlObject, final Attr attribute) throws UnmarshallingException {
        final OAuthRPExtensions descriptor = (OAuthRPExtensions) samlObject;

        if (attribute.getNamespaceURI() == null) {
            if (attribute.getLocalName().equals(OAuthRPExtensions.DEFAULT_MAX_AGE_ATTRIB_NAME)
                    && !Strings.isNullOrEmpty(attribute.getValue())) {
                descriptor.setDefaultMaxAge(Integer.parseInt(attribute.getValue()));
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.REQUIRE_AUTH_TIME_ATTRIB_NAME)
                    && !Strings.isNullOrEmpty(attribute.getValue())) {
                descriptor.setRequireAuthTime(Boolean.parseBoolean(attribute.getValue()));
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.TOKEN_ENDPOINT_AUTH_METHOD_ATTRIB_NAME)) {
                descriptor.setTokenEndpointAuthMethod(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.GRANT_TYPES_ATTRIB_NAME)) {
                descriptor.setGrantTypes(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.RESPONSE_TYPES_ATTRIB_NAME)) {
                descriptor.setResponseTypes(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.APPLICATION_TYPE_ATTRIB_NAME)) {
                descriptor.setApplicationType(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.CLIENT_URI_ATTRIB_NAME)) {
                descriptor.setClientUri(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.SCOPES_ATTRIB_NAME)) {
                descriptor.setScopes(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.SOFTWARE_ID_ATTRIB_NAME)) {
                descriptor.setSoftwareId(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.SOFTWARE_VERSION_ATTRIB_NAME)) {
                descriptor.setSoftwareVersion(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.SECTOR_IDENTIFIER_URI_ATTRIB_NAME)) {
                descriptor.setSectorIdentifierUri(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.ID_TOKEN_SIGNED_RESPONSE_ALG_ATTRIB_NAME)) {
                descriptor.setIdTokenSignedResponseAlg(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.ID_TOKEN_ENCRYPTED_RESPONSE_ALG_ATTRIB_NAME)) {
                descriptor.setIdTokenEncryptedResponseAlg(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.ID_TOKEN_ENCRYPTED_RESPONSE_ENC_ATTRIB_NAME)) {
                descriptor.setIdTokenEncryptedResponseEnc(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.USERINFO_SIGNED_RESPONSE_ALG_ATTRIB_NAME)) {
                descriptor.setUserInfoSignedResponseAlg(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.USERINFO_ENCRYPTED_RESPONSE_ALG_ATTRIB_NAME)) {
                descriptor.setUserInfoEncryptedResponseAlg(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.USERINFO_ENCRYPTED_RESPONSE_ENC_ATTRIB_NAME)) {
                descriptor.setUserInfoEncryptedResponseEnc(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.REQUEST_OBJECT_SIGNING_ALG_ATTRIB_NAME)) {
                descriptor.setRequestObjectSigningAlg(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.REQUEST_OBJECT_ENCRYPTION_ALG_ATTRIB_NAME)) {
                descriptor.setRequestObjectEncryptionAlg(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.REQUEST_OBJECT_ENCRYPTION_ENC_ATTRIB_NAME)) {
                descriptor.setRequestObjectEncryptionEnc(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.TOKEN_ENDPOINT_AUTH_SIGNING_ALG_ATTRIB_NAME)) {
                descriptor.setTokenEndpointAuthSigningAlg(attribute.getValue());
            } else if (attribute.getLocalName().equals(OAuthRPExtensions.INITIATE_LOGIN_URI_ATTRIB_NAME)) {
                descriptor.setInitiateLoginUri(attribute.getValue());
            } else {
                super.processAttribute(samlObject, attribute);
            }
        } else {
            processUnknownAttribute(descriptor, attribute);
        }
    }
// Checkstyle: CyclomaticComplexity OFF
}
