/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.encoder.impl;

import java.io.IOException;
import java.net.MalformedURLException;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletResponse;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.encoder.MessageEncoder;
import org.opensaml.messaging.encoder.MessageEncodingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.profile.config.OIDCAuthorizationConfiguration.OIDCHttpRequestMethod;
import net.shibboleth.oidc.profile.core.OIDCAuthenticationRequest;
import net.shibboleth.utilities.java.support.net.HttpServletSupport;
import net.shibboleth.utilities.java.support.net.URLBuilder;

/**
 * A {@link MessageEncoder message encoder} that encodes an OpenID authentication request by 
 * Query String Serialization and sends a HTTP redirect response.
 */
//TODO maybe this could encode authz responses as well? to replace the NimbusResponseEncoder
public class HTTPRedirectAuthnRequestEncoder extends AbstractOIDCMessageEncoder  {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(HTTPRedirectAuthnRequestEncoder.class);
    
    @Override
    public boolean test(@Nonnull final OIDCHttpRequestMethod requestMethod) {
       return requestMethod == OIDCHttpRequestMethod.GET;
    }    

    @Override
    protected void doEncode() throws MessageEncodingException {
        
        log.debug("Encoding OIDC authentication request using Query String Serialization");
        final MessageContext messageContext = getMessageContext();
        final Object outboundMessage = messageContext.getMessage();
        if (!(outboundMessage instanceof OIDCAuthenticationRequest)) {
            throw new MessageEncodingException("No outbound OIDC authentication request message "
                    + "contained in message context");
        }

        final String redirectURL = buildRedirectURL(messageContext, (OIDCAuthenticationRequest)outboundMessage);
        
        final HttpServletResponse response = getHttpServletResponse();
        HttpServletSupport.addNoCacheHeaders(response);
        HttpServletSupport.setUTF8Encoding(response);
        HttpServletSupport.setContentType(response, "application/x-www-form-urlencoded");

        try {
            log.trace("Redirecting user-agent to '{}'",redirectURL);
            response.sendRedirect(redirectURL);
        } catch (final IOException e) {
            throw new MessageEncodingException("Problem sending HTTP redirect", e);
        }
        
    }
    
    /**
     * Build the URL to redirect the client to using parameters in the authentication request.
     * 
     * @param messageContext the current message context
     * @param request the authentication request
     * 
     * @return a URL to redirect the client to.
     * 
     * @throws MessageEncodingException if there is an issue building the URL or the endpoint is null.
     */
    protected String buildRedirectURL(final MessageContext messageContext, final OIDCAuthenticationRequest request)
            throws MessageEncodingException {
        
        if (request.getEndpointURI() == null) {
            throw new MessageEncodingException("No endpoint URI specified, URL can not be built");
        }
        URLBuilder urlBuilder = null;
        try {
            if (request.getEndpointURI() == null) {
                throw new MessageEncodingException("Endpoint URL is null");
            }
            //TODO check the endpoint is always the baseURL, otherwise this may go wrong.
            urlBuilder = new URLBuilder(request.getEndpointURI().toString());
        } catch (final MalformedURLException e) {
            throw new MessageEncodingException("Endpoint URL " + request.getEndpointURI() + " is not a valid URL", e);
        }
        
        serializeAuthorizationParamsToUrl(request, urlBuilder);       
        return urlBuilder.buildURL();
        
    }
    
    

}
