/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.encoder.impl;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;

import org.opensaml.messaging.encoder.servlet.AbstractHttpServletResponseMessageEncoder;

import net.shibboleth.oidc.profile.core.OIDCAuthenticationRequest;
import net.shibboleth.oidc.profile.encoder.OIDCMessageEncoder;
import net.shibboleth.utilities.java.support.collection.Pair;
import net.shibboleth.utilities.java.support.net.URLBuilder;

/**
 * Base class for OIDC message encoders.
 */
//TODO: add a method or strategy for testing the request covers all the required parameters for a given response type 
public abstract class AbstractOIDCMessageEncoder extends AbstractHttpServletResponseMessageEncoder 
            implements OIDCMessageEncoder {
    
    
    /**
     * Serialize OAuth 2.0 authorization parameters from the authentication request to the query string 
     * of the URL.
     * 
     * @param request the authentication request.
     * @param builder the URL builder to add the query parameters to.
     */
    protected void serializeAuthorizationParamsToUrl(@Nonnull final OIDCAuthenticationRequest request,
            @Nonnull final URLBuilder builder) {
        
        final List<Pair<String, String>> params = createParametersFromRequest(request);
        params.forEach(param -> builder.getQueryParams().add(param));
    }
    
    /**
     * Serialize OAuth 2.0 authorization parameters from the authentication request to a query string.
     * 
     * @param request the authentication request query string.
     * 
     * @return the query string.
     */
    protected String serializeAuthorizationParamsToQueryString(@Nonnull final OIDCAuthenticationRequest request) {
        //TODO maybe a better way to do this than the full URL builder?
        final URLBuilder builder = new URLBuilder();
        final List<Pair<String, String>> params = createParametersFromRequest(request);
        params.forEach(param -> builder.getQueryParams().add(param));
        return builder.buildQueryString();
    }
    
    /**
     * Create a list of OAuth 2.0 authorization parameters from the {@link OIDCAuthenticationRequest} object.
     * 
     * @param req the authentication request
     * 
     * @return a list of authorization parameters.
     */
    protected List<Pair<String, String>> createParametersFromRequest(@Nonnull final OIDCAuthenticationRequest req) {
        
        final List<Pair<String, String>> params = new ArrayList<>();
        
        params.add(new Pair<>("client_id", req.getClientID().getValue()));
        params.add(new Pair<>("scope", req.getScope().toString()));
        
        if (req.getResponseType() != null) {
            params.add(new Pair<>("response_type", req.getResponseType().toString()));
        }
        if (req.getResponseMode() != null) {
            params.add(new Pair<>("response_mode", req.getResponseMode().getValue()));
        }
        if (req.getRedirectURI() != null) {
            params.add(new Pair<>("redirect_uri", req.getRedirectURI().toString()));
        }   
        if (req.getState() != null) {
            params.add(new Pair<>("state", req.getState().getValue()));
        }
        if (req.getPrompt() != null) {
            params.add(new Pair<>("prompt", req.getPrompt().toString()));
        }
        if (req.getRequestObject() != null) {          
            params.add(new Pair<>("request", req.getRequestObject().serialize()));          
        }
        if (req.getNonce() != null) {          
            params.add(new Pair<>("nonce", req.getNonce().getValue()));          
        }
        if (req.getMaxAge() != null && req.getMaxAge().toSeconds() > 0) {          
            params.add(new Pair<>("max_age", Long.toString(req.getMaxAge().toSeconds())));          
        }
        //TODO: requestURI, includedGrantedScopes?, resource_uris?
        return params;

    }

}
