/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.oauth2.config;

import java.time.Duration;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;

/**
 * Profile configuration for the OAuth2 Token Revocation.
 */
public class OAuth2TokenRevocationConfiguration extends AbstractOAuth2TokenValidatingConfiguration {

    /** OAuth2 Token Revocation URI. */
    @Nonnull @NotEmpty public static final String PROTOCOL_URI = "https://tools.ietf.org/html/rfc7009";

    /** ID for this profile configuration. */
    @Nonnull @NotEmpty public static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/oauth2/revocation";

    /** Enumeration of the OAuth2 token revocation methods. */
    public enum OAuth2TokenRevocationMethod {    
        /**
         * Revoke full chain of tokens (from authorization code (or initial access token) onwards).
         */
        CHAIN,
        /**
         * Revoke single access or refresh token.
         */
        TOKEN
    }

    /** 
     * Which revocation method should be used when revoking a token. 
     * Supported values are CHAIN and TOKEN. The default is CHAIN. 
     */
    @Nonnull private Function<ProfileRequestContext,String> revocationMethodLookupStrategy;

    /** Lookup function to supply revocation lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> revocationLifetimeLookupStrategy;

    /**
     * Constructor.
     */
    public OAuth2TokenRevocationConfiguration() {
        this(PROFILE_ID);
        revocationMethodLookupStrategy = FunctionSupport.constant(OAuth2TokenRevocationMethod.CHAIN.toString());
        revocationLifetimeLookupStrategy = FunctionSupport.constant(Duration.ofHours(6));
    }

    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public OAuth2TokenRevocationConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
    }

    /**
     * Set strategy for looking up which revocation method should be used when revoking a token.
     * 
     * @param strategy What to set.
     * 
     * @since 2.1.0
     */
    public void setRevocationMethodLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,String> strategy) {
        revocationMethodLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");        
    }

    /**
     * Set which revocation method should be used when revoking a token.
     * 
     * @param method What to set.
     * 
     * @since 2.1.0
     */
    public void setRevocationMethod(@Nonnull @NotEmpty final OAuth2TokenRevocationMethod method) {
        revocationMethodLookupStrategy = FunctionSupport.constant(method != null ? method.toString() : null);
    }

    /**
     * Get the revocation method used when revoking a token.
     * 
     * @param profileRequestContext profile request context.
     * @return The revocation method used when revoking a token.
     * 
     * @since 2.1.0
     */
    public OAuth2TokenRevocationMethod getRevocationMethod(
            @Nullable final ProfileRequestContext profileRequestContext) {
        final String method = revocationMethodLookupStrategy.apply(profileRequestContext);
        if (method != null) {
            try {
                return OAuth2TokenRevocationMethod.valueOf(method);
            } catch (final IllegalArgumentException e) {
                throw new ConstraintViolationException("Unexpected revocation method value: '" + method + "': "
                        + e.getMessage());
            }
        }
        return null;
    }

    /**
     * Set a lookup strategy for the revocation lifetime.
     *
     * @param strategy What to set.
     * 
     * @since 2.1.0
     */
    public void setRevocationLifetimeLookupStrategy(
            @Nullable final Function<ProfileRequestContext,Duration> strategy) {
        revocationLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get revocation lifetime.
     * 
     * <p>Defaults to 6 hours.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return revocation lifetime
     * 
     * @since 2.1.0
     */
    @Positive @Nonnull
    public Duration getRevocationLifetime(@Nullable final ProfileRequestContext profileRequestContext) {
        final Duration lifetime = revocationLifetimeLookupStrategy.apply(profileRequestContext);
        
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Revocation lifetime must be greater than 0");
        return lifetime;
    }
    
    /**
     * Set the lifetime of revocation.
     * 
     * @param lifetime What to set.
     * 
     * @since 2.1.0
     */
    public void setRevocationLifetime(@Positive @Nonnull final Duration lifetime) {
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Revocation lifetime must be greater than 0");
        
        revocationLifetimeLookupStrategy = FunctionSupport.constant(lifetime);
    }

}