/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.oauth2.config;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.google.common.base.Predicates;
import com.nimbusds.oauth2.sdk.GrantType;

import net.shibboleth.oidc.profile.config.AbstractOIDCSSOConfiguration;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * OIDC-aware OAuth 2 token endpoint profile configuration.
 */
public class OAuth2TokenConfiguration extends AbstractOIDCSSOConfiguration {

    /** ID for this profile configuration. */
    @Nonnull @NotEmpty public static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/oauth2/token";

    /** Enabled grant types. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> grantTypesLookupStrategy;

    /** Lookup function to supply strategy bi-function for manipulating refresh token claims set. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        refreshTokenClaimsSetManipulationStrategyLookupStrategy;

    /** Whether always revoke the refresh_token after it's used. */
    @Nonnull private Predicate<ProfileRequestContext> enforceRefreshTokenRotationPredicate;

    /**
     * Constructor.
     */
    public OAuth2TokenConfiguration() {
        this(PROFILE_ID);

        grantTypesLookupStrategy = FunctionSupport.constant(
                Set.of(GrantType.AUTHORIZATION_CODE.toString(), GrantType.REFRESH_TOKEN.toString()));
        refreshTokenClaimsSetManipulationStrategyLookupStrategy = FunctionSupport.constant(null);
        enforceRefreshTokenRotationPredicate = Predicates.alwaysFalse();
    }

    /**
     * Creates a new configuration instance.
     *
     * @param profileId unique profile identifier
     */
    public OAuth2TokenConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
    }

    /**
     * Get the enabled grant types.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return enabled grant types
     */
    @Nonnull @NonnullElements @NotLive @Unmodifiable public Set<String> getGrantTypes(
            @Nullable final ProfileRequestContext profileRequestContext) {
        
        final Collection<String> types = grantTypesLookupStrategy.apply(profileRequestContext);
        if (types != null) {
            return Set.copyOf(types);
        }
        return Collections.emptySet();
    }

    /**
     * Set the enabled grant types.
     * 
     * @param types types to enable
     */
    public void setGrantTypes(@Nonnull @NonnullElements final Collection<String> types) {
        Constraint.isNotNull(types, "Collection of types cannot be null");

        if (types != null) {
            grantTypesLookupStrategy =
                    FunctionSupport.constant(Set.copyOf(StringSupport.normalizeStringCollection(types)));
        } else {
            grantTypesLookupStrategy = FunctionSupport.constant(null);
        }
    }

    /**
     * Set a lookup strategy for the enabled grant types.
     *
     * @param strategy  lookup strategy
     */
    public void setGrantTypesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        grantTypesLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get the bi-function for manipulating refresh token claims set.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the bi-function for manipulating refresh token claims set
     * 
     * @since 2.1.0
     */
    @Nonnull
    public BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>
        getRefreshTokenClaimsSetManipulationStrategy(
            @Nullable final ProfileRequestContext profileRequestContext) {
        return refreshTokenClaimsSetManipulationStrategyLookupStrategy.apply(profileRequestContext);
    }

    /**
     * Set the bi-function for manipulating refresh token claims set.
     * 
     * @param strategy bi-function for manipulating refresh token claims set
     * 
     * @since 2.1.0
     */
    public void setRefreshTokenClaimsSetManipulationStrategy(
            @Nullable final BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> strategy) {
        refreshTokenClaimsSetManipulationStrategyLookupStrategy = FunctionSupport.constant(strategy);
    }

    /**
     * Set a lookup strategy for the bi-function for manipulating refresh token claims set.
     *
     * @param strategy lookup strategy
     * 
     * @since 2.1.0
     */
    public void setRefreshTokenClaimsSetManipulationStrategyLookupStrategy(@Nonnull final 
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        refreshTokenClaimsSetManipulationStrategyLookupStrategy = Constraint.isNotNull(strategy,
                "Lookup strategy cannot be null");
    }

    /**
     * Get whether always revoke the refresh_token after it's used.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether always revoke the refresh_token after it's used
     * 
     * @since 2.1.0
     */
    @Nonnull
    public boolean isEnforceRefreshTokenRotation(
            @Nullable final ProfileRequestContext profileRequestContext) {
        return enforceRefreshTokenRotationPredicate.test(profileRequestContext);
    }

    /**
     * Set whether always revoke the refresh_token after it's used.
     * 
     * @param flag flag to set
     * 
     * @since 2.1.0
     */
     public void setEnforceRefreshTokenRotation(final boolean flag) {
         enforceRefreshTokenRotationPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
     }

    /**
     * Set condition for whether always revoke the refresh_token after it's used.
     * 
     * @param condition condition to set
     * 
     * @since 2.1.0
     */
     public void setEnforceRefreshTokenRotationPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
         enforceRefreshTokenRotationPredicate = Constraint.isNotNull(condition, "Condition cannot be null");
     }

}